/** 
 * @file    kAccgd2kInfo.h
 * @brief   Declares the kAccgd2kInfo type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ACCGD2K_INFO_H
#define K_FIRESYNC_ACCGD2K_INFO_H

#include <kFireSync/Client/Camera/Algorithms/kAcgInfo.h>

/**
 * @class   kAccgd2kInfo
 * @extends kAcgInfo
 * @ingroup kFireSync-Client-Camera-Algorithms
 * @brief   Represents configuration for the ACCGD_2K algorithm, which provides column-wise, adaptive-threshold spot detection. 
 * 
 * Algorithm configuration options can be modified to emit results from different stages of the 
 * processing pipeline: centroids (aka 'spots'), calibrated 2D ranges, or resampled 
 * 1D ranges. Access to earlier processing stages is provided both for diagnostic/testing 
 * purposes and to support applications that require access to data from prior processing stages. 
 * However, only one type of result can be emitted at a time.
 *   
 * If range output is disabled, the algorithm will emit centroids (spots). One kMsg will be generated
 * for each camera frame and each kMsg will contain one kArrayList<kFpgaSpot>. If spot sorting is 
 * disabled, the complete list of unsorted spots (up to the configured maximum number of spots) will be 
 * delivered. In this case, the size of the delivered spot list is governed by the MaxFound property. If 
 * spot sorting is enabled, then the spot list will be reduced to one spot per column using to the configured 
 * spot sorting criteria. In this case, the size of the spot list is governed by the width of the source 
 * image.  
 * 
 * If range output is enabled, the algorithm will convert spots to calibrated range values. (Range calibration 
 * tables must be provided via the kCamera WriteRangeLut method). In this case, the algorithm will produce 
 * kProfileMsg objects, where each kProfileMsg contains results from one or more camera frames. If range 
 * resampling is disabled, the algorithm will emit kProfileMsg<kPoint16s> objects, where each element 
 * represents a 2D range value. If range resampling is enabled, the algorithm will emit kProfileMsg<k16s> 
 * objects, where each element represents a 1D range value that has been resampled to a uniform spacing along 
 * the field of view. Range processing options can be configured via methods inherited from the kAcgInfo base 
 * class. The maximum number of frames included in a single kProfileMsg can be limited via the kCameraState 
 * MaxBatchSize property. 
 */
//typedef kAcgInfo kAccgd2kInfo;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kAccgd2kInfo object.
 *
 * @public                  @memberof kAccgd2kInfo
 * @param   info            Destination for the constructed object handle. 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration id. 
 * @param   plVersion       PL configuration version.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_Construct(kAccgd2kInfo* info, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kAlloc allocator); 

/** @name EdgeThreshold */
//@{

/** 
 * Sets the spot detection threshold.
 *
 * This threshold is applied to pixel intensities to search for spot boundaries.
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   threshold   Spot detection threshold.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetEdgeThreshold(kAccgd2kInfo info, k32u threshold, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the spot detection threshold.
 *
 * This threshold is applied to pixel intensities to search for spot boundaries.
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Spot detection threshold.
 */
kFsFx(k32u) kAccgd2kInfo_EdgeThreshold(kAccgd2kInfo info); 

/** 
 * Reports constraint and validity information for the EdgeThreshold setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_EdgeThresholdInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name MinWidth */
//@{

/** 
 * Sets the minimum spot width threshold.
 *
 * Spots with less than MinWidth pixels are excluded from output. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   minWidth    Minimum spot width, in pixels.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetMinWidth(kAccgd2kInfo info, k32u minWidth, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the minimum spot width threshold.
 *
 * Spots with less than MinWidth pixels are excluded from output. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Minimum spot width, in pixels.
 */
kFsFx(k32u) kAccgd2kInfo_MinWidth(kAccgd2kInfo info); 

/** 
 * Reports constraint and validity information for the MinWidth setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_MinWidthInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name MaxWidth */
//@{

/** 
 * Sets the maximum spot width threshold.
 *
 * Spots with more than MaxWidth pixels are excluded from output. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   maxWidth    Maximum spot width threshold, in pixels.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetMaxWidth(kAccgd2kInfo info, k32u maxWidth, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the maximum spot width threshold.
 *
 * Spots with more than MaxWidth pixels are excluded from output. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Maximum spot width threshold, in pixels.   
 */
kFsFx(k32u) kAccgd2kInfo_MaxWidth(kAccgd2kInfo info); 

/** 
 * Reports constraint and validity information for the MaxWidth setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_MaxWidthInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name MinSum */
//@{

/** 
 * Sets the minimum spot sum threshold.
 *
 * Spots with sum values less than this threshold are excluded from output.
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   minSum      Minimum spot sum threshold.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetMinSum(kAccgd2kInfo info, k32u minSum, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the minimum spot sum threshold.
 *
 * Spots with sum values less than this threshold are excluded from output.
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Minimum spot sum threshold.
 */
kFsFx(k32u) kAccgd2kInfo_MinSum(kAccgd2kInfo info); 

/** 
 * Reports constraint and validity information for the MinSum setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_MinSumInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name MaxFound */
//@{

/** 
 * Sets the maximum number of spots that can be detected. 
 *
 * If the specified maximum number of spots are found, any additional spots will be omitted 
 * from output. 
 * 
 * If spot sorting is disabled, the MaxFound parameter determines the size of the data array that 
 * will be delivered from PL to PS memory, regardless of the number of spots actually found 
 * (empty space at the end of the list is filled with special null values). Accordingly, to avoid 
 * wasting controller memory bandwidth, the MaxFound parameter should normally be set to the smallest 
 * viable value when receiving unsorted spots. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   maxFound    Maximum number of spots that can be detected.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetMaxFound(kAccgd2kInfo info, k32u maxFound, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the maximum number of spots that can be detected. 
 *
 * If the specified maximum number of spots are found, any additional spots will be omitted 
 * from output. 
 * 
 * If spot sorting is disabled, the MaxFound parameter determines the size of the data array that 
 * will be delivered from PL to PS memory, regardless of the number of spots actually found 
 * (empty space at the end of the list is filled with special null values). Accordingly, to avoid 
 * wasting controller memory bandwidth, the MaxFound parameter should normally be set to the smallest 
 * viable value when receiving unsorted spots. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Maximum number of spots that can be detected.
 */
kFsFx(k32u) kAccgd2kInfo_MaxFound(kAccgd2kInfo info); 

/** 
 * Reports constraint and validity information for the MaxFound setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_MaxFoundInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name SpotSortEnabled */
//@{

/** 
 * Enables or disables spot sorting.
 *
 * If spot sorting is enabled, spots will be sorted into slices (columns) and filtered 
 * such that there is one spot per slice. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Enable spot sorting?   
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_EnableSpotSort(kAccgd2kInfo info, kBool enabled);

/** 
 * Reports whether spot sorting is enabled.
 * 
 * If spot sorting is enabled, spots will be sorted into slices (columns) and filtered 
 * such that there is one spot per slice. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if sorting is enabled.
 */
kFsFx(kBool) kAccgd2kInfo_SpotSortEnabled(kAccgd2kInfo info);

/** 
 * Reports constraint and validity information for the EnableSpotSort setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SpotSortEnabledInfo(kAccgd2kInfo info, kInfoBool* infoBool);

//@}
/** @name SpotSortInverted */
//@{

/** 
 * Enables or disables reverse spot order.
 * 
 * The default sorting direction yields ascending slice indices in algorithm output data.
 * The SpotSortInverted option can be used to reverse the spot output order. 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   invert      Enable reverse spot-sorting?
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_InvertSpotSort(kAccgd2kInfo info, kBool invert);

/** 
 * Reports whether spot sorting order is reversed.
 * 
 * The default sorting direction yields ascending slice indices in algorithm output data.
 * The SpotSortInverted option can be used to reverse the spot output order. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Is reverse spot sorting enabled?
 */
kFsFx(kBool) kAccgd2kInfo_SpotSortInverted(kAccgd2kInfo info);

/** 
 * Reports constraint and validity information for the SpotSortInverted setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_InvertSpotSortInfo(kAccgd2kInfo info, kInfoBool* infoBool);

//@}
/** @name SpotSortType */
//@{

/** 
 * Sets the spot sorting methodology.
 * 
 * When spot sorting is enabled, SpotSortType determines the criterion used to map elements of the 
 * unsorted spot array, which may include multiple spots per slice (column), to indices within the 
 * sorted output array (one spot per slice). 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   type        Sorting methodology.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SetSpotSortType(kAccgd2kInfo info, kCameraSpotSort type);

/** 
 * Gets the spot sorting methodology.
 * 
 * When spot sorting is enabled, SpotSortType determines the criterion used to map elements of the 
 * unsorted spot array, which may include multiple spots per slice (column), to indices within the 
 * sorted output array (one spot per slice). 
 * 
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @return              Sorting methodology.
 */
kFsFx(kCameraSpotSort) kAccgd2kInfo_SpotSortType(kAccgd2kInfo info);

/** 
 * Reports constraint and validity information for the SpotSortType setting. 
 *
 * @public              @memberof kAccgd2kInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kAccgd2kInfo_SpotSortTypeInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}

#include <kFireSync/Client/Camera/Algorithms/kAccgd2kInfo.x.h>

#endif
