/**
* @file    GvRangeMsg.h
* @brief   Declares the GvRangeMsg class.
*
* Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GV_RANGE_MSG_H
#define GV_RANGE_MSG_H

#include <GoVision/Data/GvDataMsg.h>

/**
* Encapsulates a profile message.
*
* @class       GvRangeMsg
* @extends     GvDataMsg
* @ingroup     GoVision-Data
*/
typedef GvDataMsg GvRangeMsg;

GvFx(kStatus) GvRangeMsg_Construct(GvRangeMsg* msg, kType pointType, kSize count, kSize pointCount, kAlloc alloc);

/** 
 * Copies elementary range data from the source frame.
 * This should be used when decimation is applicable and the frames are being explicitly handled.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   msgIndex    Message frame index.
 * @param   source      Source message object.
 * @param   sourceIndex Source message frame index.
 * @return              Operation status.
 */
GvFx(kStatus) GvRangeMsg_CopyAttrs(GvRangeMsg msg, kSize msgIndex, GvRangeMsg source, kSize sourceIndex);

/** 
 * Returns the message point type.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Point type.
 */
GvFx(kType) GvRangeMsg_PointType(GvRangeMsg msg);

/** 
 * Returns the number of points.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Point count.
 */
GvFx(kSize) GvRangeMsg_PointCount(GvRangeMsg msg);

/** 
 * Returns the x,y,z offset of the data message.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Pointer to data offset.
 */
GvFx(const kPoint3d64f*) GvRangeMsg_Offset(GvRangeMsg msg);

/** 
 * Sets the x,y,z offset of the data messae.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   offset      Pointer to data offset.
 * @return              Operation status.
 */
GvFx(kStatus) GvRangeMsg_SetOffset(GvRangeMsg msg, const kPoint3d64f* offset);

/** 
 * Returns the x,y,z scale of the data message.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Pointer to data scale.
 */
GvFx(const kPoint3d64f*) GvRangeMsg_Scale(GvRangeMsg msg);

/** 
 * Sets the x,y,z scale of the data message.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   scale       Pointer to data scale.
 * @return              Operation status.
 */
GvFx(kStatus) GvRangeMsg_SetScale(GvRangeMsg msg, const kPoint3d64f* scale);

/** 
 * Returns a pointer to the data at the index.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   frameIndex  Frame index. 
 * @return              Pointer to data at frame index.
 */
GvFx(void*) GvRangeMsg_Point(GvRangeMsg msg, kSize frameIndex);

/** 
 * Returns whether or not this message contains intensity data.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Presence of intensity data.
 */
GvFx(kBool) GvRangeMsg_HasIntensity(GvRangeMsg msg);

/** 
 * Returns the message intensity type.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Intensity type.
 */
GvFx(kType) GvRangeMsg_IntensityType(GvRangeMsg msg);

/** 
 * Allocates an intensity buffer for the message.
 *
 * If the buffer already exists and is of the same type as requested, success
 * is returned. If it exists but type does not match, kERROR_ALREADY_EXISTS is returned.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   type        Type of intensity values.
 * @return              Operation status.
 */
GvFx(kStatus) GvRangeMsg_AllocateIntensity(GvRangeMsg msg, kType type);

/** 
 * Returns a pointer to the intensity buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @param   frameIndex  Frame index. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvRangeMsg_Intensity(GvRangeMsg msg, kSize frameIndex);

/** 
 * Returns a pointer to the slices buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvRangeMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvRangeMsg_Slice(GvRangeMsg msg);

/**
 * Sets the rendering object within the message.
 *
 * @public               @memberof GvRangeMsg
 * @param    msg         Message object.
 * @param    object      GdkGraphic object (ownership is transferred).
 * @return               Operation status.
 */
GvFx(kStatus) GvRangeMsg_SetRendering(GvRangeMsg msg, kObject object);

/**
 * Retrieves the rendering object from the message.
 *
 * @public               @memberof GvRangeMsg
 * @param    msg         Message object.
 * @return               GdkGraphic object (ownership is transferred).
 */
GvFx(kObject) GvRangeMsg_Rendering(GvRangeMsg msg);

#include <GoVision/Data/GvRangeMsg.x.h>

#endif
