/**
 * @file    TestMesh.cpp
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestMesh.h>

kBeginClassEx(Tool, TestMesh)
    kAddVMethod(TestMesh, kObject, VRelease)
    kAddVMethod(TestMesh, GdkTool, VInit)
    kAddVMethod(TestMesh, GdkTool, VName)
    kAddVMethod(TestMesh, GdkTool, VDescribe)
    kAddVMethod(TestMesh, GdkTool, VCalcDataOutputRegionInstanced)
    kAddVMethod(TestMesh, GdkTool, VUpdateConfigInstanced)
    kAddVMethod(TestMesh, GdkTool, VStart)
    kAddVMethod(TestMesh, GdkTool, VProcess)
kEndClassEx()

#define DEFAULT_RADIUS                  (8)
#define DEFAULT_LATITUDE_COUNT          (20)
#define DEFAULT_LONGITUDE_COUNT         (20)

ToolFx(const kChar*) TestMesh_VName()
{
    return TEST_MESH_TOOL_NAME;
}

ToolFx(kStatus) TestMesh_VDescribe(GdkToolInfo toolInfo)
{
    GdkParamsInfo paramsInfo;
    GdkParamInfo paramInfo;
    GdkToolDataOutputInfo dataInfo;

    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_MESH_TOOL_NAME));
    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_MESH_TOOL_LABEL_FEATURE));
    kCheck(GdkToolInfo_SetIsPrivate(toolInfo, kTRUE));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_FEATURES));

    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    paramsInfo = GdkToolInfo_Params(toolInfo);

    kCheck(GdkParamsInfo_AddFloat(paramsInfo, "radius", "Radius", DEFAULT_RADIUS, &paramInfo));
    kCheck(GdkParamsInfo_AddInt(paramsInfo, "latitudeCount", "Latitude Count", DEFAULT_LATITUDE_COUNT, &paramInfo));
    kCheck(GdkParamInfo_SetMinInt(paramInfo, DEFAULT_LATITUDE_COUNT));
    kCheck(GdkParamsInfo_AddInt(paramsInfo, "longitudeCount", "Longitude Count", DEFAULT_LONGITUDE_COUNT, &paramInfo));
    kCheck(GdkParamInfo_SetMinInt(paramInfo, DEFAULT_LONGITUDE_COUNT));
    
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MESH, "MeshOutput", "MeshOutput", &dataInfo));
    kCheck(GdkToolDataOutputInfo_SetHasIntensity(dataInfo, kTRUE));

    return kOK;
}

ToolFx(kStatus) TestMesh_VInit(TestMesh tool, kType type, kAlloc alloc)
{
    kObjR(TestMesh, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));

    obj->latitudeCount = 0;
    obj->longitudeCount = 0;

    return kOK;
}

ToolFx(kStatus) TestMesh_VRelease(TestMesh tool)
{
    kObj(TestMesh, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestMesh_VCalcDataOutputRegionInstanced(GdkTool tool, GdkToolDataOutputCfg outputConfig, GdkRegion3d64f* region)
{    
    region->x = -50;
    region->y = -50;
    region->z = -50;
    region->width = 100;
    region->length = 100;
    region->height = 100;

    return kOK;
}

ToolFx(kStatus) TestMesh_VUpdateConfigInstanced(TestMesh tool, GdkToolCfg toolConfig)
{

    GdkParams params = GdkToolCfg_Parameters(toolConfig);
    GdkParam param = GdkParams_Find(params, "latitudeCount");
    k32s latitudeCount = GdkParam_AsInt(param);

    if (latitudeCount % 2 == 0)
    {
        latitudeCount++;
        GdkParam_SetInt(param, latitudeCount);
    }
                                        
    return kOK;
}

ToolFx(kStatus) TestMesh_VStart(TestMesh tool)
{
    kObj(TestMesh, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkParam param;

    param = GdkParams_Find(params, "radius");
    obj->radius = GdkParam_AsFloat(param);

    param = GdkParams_Find(params, "longitudeCount");
    obj->longitudeCount = GdkParam_AsInt(param);

    param = GdkParams_Find(params, "latitudeCount");
    obj->latitudeCount = GdkParam_AsInt(param);

    obj->facetCount = obj->longitudeCount * (obj->latitudeCount - 1) * 2;

    return kOK;
}

ToolFx(kStatus) TestMesh_VProcess(TestMesh tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestMesh, tool);

    kTry
    {
        GvMeshMsg meshMsg = kNULL;
        
        kTest(GdkToolOutput_InitMeshAt(output, 0, &meshMsg));

        if (!kIsNull(meshMsg))
        {
            kTest(TestMesh_GenerateUvSphere(tool, meshMsg));

            // Note: Normal data calculation is optional. 
            // If normals are not calculated and set by the tool, they will be calculated by the UI at the cost of client side performance.
            kTest(TestMesh_CalculateNormals(tool, meshMsg));
            kTest(TestMesh_AddCurvature(tool, meshMsg));
            kTest(TestMesh_AddTexture(tool, meshMsg));
        }
    }
    kFinally
    {
        kEndFinally();
    }

    return kOK;
}

ToolFx(kStatus) TestMesh_GenerateUvSphere(TestMesh tool, GvMeshMsg msg)
{
    kObj(TestMesh, tool);
    kObjN(GvMeshMsg, meshMsg, msg);

    kPoint3d32f* vertexData;
    kSize vertexCount = obj->longitudeCount * obj->latitudeCount + 2;
    kPoint3d64f center = {0.0, 0.0, 0.0};
    k64f radius = obj->radius;
    k64f phiSlice = kMATH_PI / (obj->latitudeCount + 1);
    k64f thetaSlice = 2.0 * kMATH_PI / obj->longitudeCount;
    kSize index = 0;

    kCheck(GvMeshMsg_AllocateVertexData(meshMsg, vertexCount, GdkTool_MessageAlloc(tool)));
    kArray1 vertex = GvMeshMsg_VertexData(meshMsg);
    kCheck(GvMeshMsg_SetVertexDataCount(meshMsg, vertexCount));

    // Add UV vertices to the sphere
    for (kSize i = 0; i < obj->latitudeCount; i++)
    {
        k64f phi = phiSlice * (i + 1);
        for (kSize j = 0; j < obj->longitudeCount; j++)
        {
            k64f theta = thetaSlice * j;
            index = i * obj->longitudeCount + j;
            kAssert(index < vertexCount);
            vertexData = kArray1_DataAtT(vertex, index, kPoint3d32f);
            vertexData->x = (k32f)(center.x + radius * sin(phi) * cos(theta));
            vertexData->y = (k32f)(center.y + radius * sin(phi) * sin(theta));
            vertexData->z = (k32f)(center.z + radius * cos(phi));
        }
    }

    if (obj->facetCount > 0)
    {
        kCheck(GvMeshMsg_AllocateFacetData(meshMsg, obj->facetCount, GdkTool_MessageAlloc(tool)));
        kArray1 facet = GvMeshMsg_FacetData(meshMsg);
        GvFacet32u* facetData;
        kCheck(GvMeshMsg_SetFacetDataCount(meshMsg, obj->facetCount));

        // Create UV Map Facets
        index = 0;
        for (kSize i = 0; i < obj->latitudeCount - 1; i++)
        {
            for (kSize j = 0; j < obj->longitudeCount; j++)
            {                
                if (j == obj->longitudeCount - 1)
                {
                    kAssert(index < obj->facetCount);
                    facetData = kArray1_DataAtT(facet, index, GvFacet32u);
                    facetData->vertex1 = (k32u)((i    ) * obj->longitudeCount + (j    ));
                    facetData->vertex2 = (k32u)((i + 1) * obj->longitudeCount + (j    ));
                    facetData->vertex3 = (k32u)((i    ) * obj->longitudeCount + (j + 1)); 
                    index++;

                    kAssert(index < obj->facetCount);
                    facetData = kArray1_DataAtT(facet, index, GvFacet32u);
                    facetData->vertex1 = (k32u)( i      * obj->longitudeCount + (j    ));
                    facetData->vertex2 = (k32u)((i    ) * obj->longitudeCount + (j + 1));
                    facetData->vertex3 = (k32u)((i - 1) * obj->longitudeCount + (j + 1));   
                    index++;
                }
                else
                {
                    kAssert(index < obj->facetCount);
                    facetData = kArray1_DataAtT(facet, index, GvFacet32u);
                    facetData->vertex1 = (k32u)((i    ) * obj->longitudeCount + (j    ));
                    facetData->vertex2 = (k32u)((i + 1) * obj->longitudeCount + (j    ));
                    facetData->vertex3 = (k32u)((i + 1) * obj->longitudeCount + (j + 1)); 
                    index++;

                    kAssert(index < obj->facetCount);
                    facetData = kArray1_DataAtT(facet, index, GvFacet32u);
                    facetData->vertex1 = (k32u)((i    ) * obj->longitudeCount + (j    ));
                    facetData->vertex2 = (k32u)((i + 1) * obj->longitudeCount + (j + 1));
                    facetData->vertex3 = (k32u)((i    ) * obj->longitudeCount + (j + 1)); 
                    index++;
                }
            }
        }
    }

    return kOK;
}

ToolFx(kStatus) TestMesh_CalculateNormals(TestMesh tool, GvMeshMsg meshMsg)
{
    kObj(TestMesh, tool);
    kArray1 facetNormal = kNULL;
    kArray1 vertexNormal = kNULL;
    GvFacet32u* facetData = kNULL;
    kPoint3d32f* facetNormalData = kNULL;
    kPoint3d32f* vertexNormalData = kNULL;
    kArray1 facet = GvMeshMsg_FacetData(meshMsg);
    kArray1 vertex = GvMeshMsg_VertexData(meshMsg);
    kPoint3d32f *vertex1, *vertex2, *vertex3;
    kSize facetNormalSize = GvMeshMsg_AllocatedFacetNormalDataCount(meshMsg);
    kSize vertexSize = GvMeshMsg_AllocatedVertexDataCount(meshMsg);

    kCheck(GvMeshMsg_AllocateFacetNormalData(meshMsg, GdkTool_MessageAlloc(tool)));
    facetNormal = GvMeshMsg_FacetNormalData(meshMsg);
    kCheck(GvMeshMsg_SetFacetNormalDataCount(meshMsg, obj->facetCount));

    // Facet normal calculation is required to be able to calculate Vertex Normals
    for (kSize i = 0; i < kArray1_Count(facetNormal); i++)
    {
        facetNormalData = kArray1_DataAtT(facetNormal, i, kPoint3d32f);

        facetData = kArray1_DataAtT(facet, i, GvFacet32u);
        vertex1 = kArray1_DataAtT(vertex, facetData->vertex1, kPoint3d32f);
        vertex2 = kArray1_DataAtT(vertex, facetData->vertex2, kPoint3d32f);
        vertex3 = kArray1_DataAtT(vertex, facetData->vertex3, kPoint3d32f);

        kCheck(TestMesh_CalculateFacetNormal(vertex1, vertex2, vertex3, facetNormalData));
    }

    if (facetNormalSize > 0)
    {
        kCheck(GvMeshMsg_AllocateVertexNormalData(meshMsg, GdkTool_MessageAlloc(tool)));
        vertexNormal = GvMeshMsg_VertexNormalData(meshMsg);
        kCheck(GvMeshMsg_SetVertexNormalDataCount(meshMsg, vertexSize));

        //First need to accumulate all facet normal according to its vertex
        for (kSize i = 0; i < kArray1_Count(facetNormal); i++)
        {
            facetNormalData = kArray1_DataAtT(facetNormal, i, kPoint3d32f);

            facetData = kArray1_DataAtT(facet, i, GvFacet32u);
            vertex1 = kArray1_DataAtT(vertexNormal, facetData->vertex1, kPoint3d32f);
            vertex2 = kArray1_DataAtT(vertexNormal, facetData->vertex2, kPoint3d32f);
            vertex3 = kArray1_DataAtT(vertexNormal, facetData->vertex3, kPoint3d32f);

            kCheck(TestMesh_Add3d32f(vertex1, facetNormalData, vertex1));
            kCheck(TestMesh_Add3d32f(vertex2, facetNormalData, vertex2));
            kCheck(TestMesh_Add3d32f(vertex3, facetNormalData, vertex3));
        }

        //Normalize sum of facet normal over a vertex
        for (kSize i = 0; i < kArray1_Count(vertexNormal); i++)
        {
            vertexNormalData = kArray1_DataAtT(vertexNormal, i, kPoint3d32f);
            kCheck(TestMesh_Normalize3d32f(vertexNormalData));
        }
    }

    return kOK;
}

ToolFx(kStatus) TestMesh_AddCurvature(TestMesh tool, GvMeshMsg meshMsg)
{
    kCheck(GvMeshMsg_AllocateVertexCurvatureData(meshMsg, GdkTool_MessageAlloc(tool)));
    kSize vertexSize = GvMeshMsg_AllocatedVertexDataCount(meshMsg);
    kArray1 vertexCurvature = GvMeshMsg_VertexCurvatureData(meshMsg);
    k32f* vertexCurvatureData;

    for (kSize i = 0; i < kArray1_Count(vertexCurvature); i++)
    {
        vertexCurvatureData = kArray1_DataAtT(vertexCurvature, i, k32f);
        *vertexCurvatureData = (k32f)((i + 1) * (i + 1) + (i + 1) * 1.5);
    }
    kCheck(GvMeshMsg_SetVertexCurvatureDataCount(meshMsg, vertexSize));
    return kOK;
}

ToolFx(kStatus) TestMesh_AddTexture(TestMesh tool, GvMeshMsg meshMsg)
{   
    kCheck(GvMeshMsg_AllocateVertexTextureData(meshMsg, GdkTool_MessageAlloc(tool)));
    kSize vertexSize = GvMeshMsg_AllocatedVertexDataCount(meshMsg);
    kArray1 vertexTexture = GvMeshMsg_VertexTextureData(meshMsg);
    k8u* vertexTextureData;

    for (kSize i = 0; i < kArray1_Count(vertexTexture); i++)
    {
        vertexTextureData = kArray1_DataAtT(vertexTexture, i, k8u);
        *vertexTextureData = (k8u)((k32f)k8U_MAX / vertexSize * i);
    }
    kCheck(GvMeshMsg_SetVertexTextureDataCount(meshMsg, vertexSize));
        
    return kOK;
}

ToolFx(kStatus) TestMesh_Add3d32f(kPoint3d32f *p1, kPoint3d32f *p2, kPoint3d32f *sum)
{
    sum->x = p1->x + p2->x;
    sum->y = p1->y + p2->y;
    sum->z = p1->z + p2->z;

    return kOK;
}

ToolFx(kStatus) TestMesh_Sub3d32f(kPoint3d32f *p1, kPoint3d32f *p2, kPoint3d32f *diff)
{
    diff->x = p2->x - p1->x;
    diff->y = p2->y - p1->y;
    diff->z = p2->z - p1->z;

    return kOK;
}

ToolFx(kStatus) TestMesh_CrossProduct3d32f(kPoint3d32f *p1, kPoint3d32f *p2, kPoint3d32f *cproduct)
{
    cproduct->x = (p1->y * p2->z) - (p1->z * p2->y);
    cproduct->y = (p1->z * p2->x) - (p1->x * p2->z);
    cproduct->z = (p1->x * p2->y) - (p1->y * p2->x);

    return kOK;
}

ToolFx(kStatus) TestMesh_Normalize3d32f(kPoint3d32f *vector)
{
    k32f length = (k32f)sqrt(vector->x * vector->x + vector->y * vector->y + vector->z * vector->z);
    vector->x /= length;
    vector->y /= length;
    vector->z /= length;

    return kOK;
}

ToolFx(kStatus) TestMesh_CalculateFacetNormal(kPoint3d32f *p1, kPoint3d32f *p2, kPoint3d32f *p3, kPoint3d32f *normal)
{
    kPoint3d32f v1, v2;

    kCheck(TestMesh_Sub3d32f(p1, p2, &v1));
    kCheck(TestMesh_Sub3d32f(p1, p3, &v2));

    kCheck(TestMesh_CrossProduct3d32f(&v1, &v2, normal));
    kCheck(TestMesh_Normalize3d32f(normal));

    return kOK;
}
