/////////////////////////////////////////////////////////////////////////////
// TestGenericInput
// 2018-12-12 (C) LMI Technologies
//
// Tool to test the usage of generic input.
/////////////////////////////////////////////////////////////////////////////

#include <GdkAppSample/TestGenericInput.h>
#include <GdkAppSample/GdkPartSegmentResult.h>
#include <vector>

kBeginClassEx(Tool, TestGenericInput)
    kAddVMethod(TestGenericInput, kObject, VRelease)
    kAddVMethod(TestGenericInput, GdkTool, VInit)
    kAddVMethod(TestGenericInput, GdkTool, VName)
    kAddVMethod(TestGenericInput, GdkTool, VDescribe)
    kAddVMethod(TestGenericInput, GdkTool, VStart)
    kAddVMethod(TestGenericInput, GdkTool, VStop)
    kAddVMethod(TestGenericInput, GdkTool, VProcess)
kEndClassEx()

using namespace std;

kStatus TestGenericInput_OutputValueAndPoints(kObject tool, kSize index, k64f value, kBool valid,
    vector<kPoint3d32f> centerPointList, vector<kPoint3d32f> rectanglePointList, vector<k64f> partAreaList,
    kSize highlight, GdkToolOutput output);
kStatus TestGenericInput_SendFeaturePoint(TestGenericInput tool, GdkToolOutput output, kSize index, const kPoint3d64f* point);

kSize TEST_GENERIC_INPUT_OUTPUT_FEATURE_START = 0;
kSize TEST_GENERIC_INPUT_MEASUREMENT_COUNT = 0;
kSize TEST_GENERIC_INPUT_FEATURE_COUNT = 0;

/////////////////////////////////////////////////////////////////////////////
// GtTool functions
/////////////////////////////////////////////////////////////////////////////

ToolFx(const kChar*) TestGenericInput_VName()
{
    return TEST_GENERIC_INPUT_TOOL_NAME;
}

ToolFx(kStatus) TestGenericInput_VDescribe(GdkToolInfo toolInfo)
{
    GdkParamsInfo paramsInfo = GdkToolInfo_Params(toolInfo);
    GdkParamInfo paramInfo;
    GdkFeatureInfo featureInfo = kNULL;
    GdkMeasurementInfo mmtInfo;

    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_GENERIC_INPUT_TOOL_LABEL));
    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_GENERIC_INPUT_TOOL_NAME));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    kCheck(GdkToolInfo_AddInput(toolInfo, GDK_PART_SEGEMENT_DATA_TYPE, TEST_GENERIC_INPUT_INPUT, TEST_GENERIC_INPUT_INPUT_LABEL, &paramInfo));
    kCheck(GdkParamInfo_SetIsOptional(paramInfo, kTRUE));

    for (kSize i = 0; i < TEST_GENERIC_INPUT_MEASUREMENTS_PARTS; ++i)
    {
        kChar paramName[48];
        kChar paramLabel[48];
        kCheck(kStrPrintf(paramName,  48, TEST_GENERIC_INPUT_PARAM_INDEX,       i));
        kCheck(kStrPrintf(paramLabel, 48, TEST_GENERIC_INPUT_PARAM_INDEX_LABEL, i+1));

        kCheck(GdkParamsInfo_AddInt(paramsInfo, paramName, paramLabel, (k32s)i, &paramInfo));
        kCheck(GdkParamInfo_SetIsOptional(paramInfo, kTRUE));
    }

    kCheck(GdkParamsInfo_AddBool(paramsInfo, TEST_GENERIC_INPUT_PARAM_DETAILS, TEST_GENERIC_INPUT_PARAM_DETAILS_LABEL, kFALSE, &paramInfo));
    kCheck(GdkParamInfo_SetIsOptional(paramInfo, kTRUE));

    for (kSize i = 1; i <= TEST_GENERIC_INPUT_MEASUREMENTS_PARTS; ++i)
    {
        kChar xName[48];
        kChar xLabel[48];
        kChar yName[48];
        kChar yLabel[48];
        kChar widthName[48];
        kChar widthLabel[48];
        kChar lengthName[48];
        kChar lengthLabel[48];

        kCheck(kStrPrintf(xName,        48, TEST_GENERIC_INPUT_MEASUREMENT_X,              i));
        kCheck(kStrPrintf(xLabel,       48, TEST_GENERIC_INPUT_MEASUREMENT_X_LABEL,        i));
        kCheck(kStrPrintf(yName,        48, TEST_GENERIC_INPUT_MEASUREMENT_Y,              i));
        kCheck(kStrPrintf(yLabel,       48, TEST_GENERIC_INPUT_MEASUREMENT_Y_LABEL,        i));
        kCheck(kStrPrintf(widthName,    48, TEST_GENERIC_INPUT_MEASUREMENT_WIDTH,          i));
        kCheck(kStrPrintf(widthLabel,   48, TEST_GENERIC_INPUT_MEASUREMENT_WIDTH_LABEL,    i));
        kCheck(kStrPrintf(lengthName,   48, TEST_GENERIC_INPUT_MEASUREMENT_LENGTH,         i));
        kCheck(kStrPrintf(lengthLabel,  48, TEST_GENERIC_INPUT_MEASUREMENT_LENGTH_LABEL,   i));

        kCheck(GdkToolInfo_AddMeasurement(toolInfo, xName, &mmtInfo));
        kCheck(GdkMeasurementInfo_SetLabel(mmtInfo, xLabel));
        kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_X));

        kCheck(GdkToolInfo_AddMeasurement(toolInfo, yName, &mmtInfo));
        kCheck(GdkMeasurementInfo_SetLabel(mmtInfo, yLabel));
        kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Y));

        kCheck(GdkToolInfo_AddMeasurement(toolInfo, widthName, &mmtInfo));
        kCheck(GdkMeasurementInfo_SetLabel(mmtInfo, widthLabel));
        kCheck(GdkToolInfo_AddMeasurement(toolInfo, lengthName, &mmtInfo));
        kCheck(GdkMeasurementInfo_SetLabel(mmtInfo, lengthLabel));

        // Output Features
        kChar featureName[48];
        kChar featureLabel[48];
        kCheck(kStrPrintf(featureName, 48, TEST_GENERIC_INPUT_FEATURE_CENTER, i));
        kCheck(kStrPrintf(featureLabel, 48, TEST_GENERIC_INPUT_FEATURE_CENTER_LABEL, i));

        kCheck(GdkToolInfo_AddFeature(toolInfo, featureName, GDK_FEATURE_TYPE_POINT, &featureInfo));
        kCheck(GdkFeatureInfo_SetLabel(featureInfo, featureLabel));
    }

    TEST_GENERIC_INPUT_MEASUREMENT_COUNT = 4 * TEST_GENERIC_INPUT_MEASUREMENTS_PARTS;
    TEST_GENERIC_INPUT_FEATURE_COUNT = TEST_GENERIC_INPUT_MEASUREMENTS_PARTS;

    TEST_GENERIC_INPUT_OUTPUT_FEATURE_START = 0;

    return kOK;
}

ToolFx(kStatus) TestGenericInput_VInit(TestGenericInput tool, kType type, kAlloc alloc)
{
    kObjR(TestGenericInput, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    kZero(obj->segIndices);
    obj->showDetails = kFALSE;

    return kOK;
}

ToolFx(kStatus) TestGenericInput_VRelease(TestGenericInput tool)
{
    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestGenericInput_VStop(TestGenericInput tool)
{
    kObj(TestGenericInput, tool);
    kDestroyRef(&obj->segIndices);
    return kOK;
}

ToolFx(kStatus) TestGenericInput_VStart(TestGenericInput tool)
{
    kObj(TestGenericInput, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkParam param;

    obj->dataSource = GdkToolCfg_Source(config);

    kCheck(kArrayList_Construct(&obj->segIndices, kTypeOf(kSize), TEST_GENERIC_INPUT_MEASUREMENTS_PARTS, GdkTool_MessageAlloc(tool)));
    for (kSize i = 0; i < TEST_GENERIC_INPUT_MEASUREMENTS_PARTS; ++i)
    {
        kChar paramName[48];
        kCheck(kStrPrintf(paramName, 48, TEST_GENERIC_INPUT_PARAM_INDEX, i));

        kSize segIndex = (kSize)GdkParam_AsInt(GdkParams_Find(params, paramName));
        kCheck(kArrayList_Add(obj->segIndices, &segIndex));
    }

    param = GdkParams_Find(params, TEST_GENERIC_INPUT_PARAM_DETAILS);
    obj->showDetails = GdkParam_AsBool(param);

    return kOK;
}

ToolFx(kStatus) TestGenericInput_VProcess(TestGenericInput tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestGenericInput, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkInputItem item;
    kPoint3d32f pt = {0,0,0};
    kStatus e = kOK;
    vector<kPoint3d32f> centerPoints;
    vector<kPoint3d32f> bBoxLines;
    vector<k64f>        areas;

    GvSurfaceMsg surfaceMsg = kNULL;
    GvGenericMsg arrayMsg   = kNULL;

    kTry
    {
        item = GdkToolInput_Find(input, obj->dataSource);
        if (!item)
            kThrow(kERROR_PARAMETER);

        // Get the Surface and the generic IO array
        kTest(GdkToolInput_InputDataAt(input, GDK_DATA_PRIMARY_INPUT_INDEX, obj->dataSource, &surfaceMsg));
        kTest(GdkToolInput_InputDataAt(input, GDK_DATA_SECONDARY_INPUT_START_INDEX, obj->dataSource, &arrayMsg));

        if(!surfaceMsg || !arrayMsg)
            kThrow(kERROR_PARAMETER);
        if (GvGenericMsg_IsObject(arrayMsg)  && kIsNull(GvGenericMsg_Object(arrayMsg)))
            kThrow(kERROR_PARAMETER);
        if (!GvGenericMsg_IsObject(arrayMsg) && kIsNull(GvGenericMsg_Buffer(arrayMsg)))
            kThrow(kERROR_PARAMETER);

        // Extract the parts vector
        kSize dataSize = GvGenericMsg_BufferSize(arrayMsg);
        std::vector<GdkPartSegmentResultClass> parts;
        kTest(GdkPartSegmentResult_DeserializeParts(parts, GvGenericMsg_Buffer(arrayMsg), dataSize, GdkTool_MessageAlloc(tool)));
        kSize count = parts.size();

        for (auto part : parts)
        {
            if (!part.contourPoints.empty())
            {
                // Center points
                kPoint3d_Init_(&pt, part.center.x, part.center.y, (k32f)part.heightMax);
                centerPoints.push_back(pt);

                // Bboxes
                for (int i = 0; i < TEST_GENERIC_INPUT_POINT_COUNT_PER_PART; ++i)
                {
                    kPoint3d_Init_(&pt, part.minAreaRectCorners[i % 4].x, part.minAreaRectCorners[i % 4].y, (k32f)part.heightMax);
                    bBoxLines.push_back(pt);
                }

                // Areas
                areas.push_back(part.area);
            }
        }

        kSize measIdx = 0;
        for (kSize partIdx = 0; partIdx < TEST_GENERIC_INPUT_MEASUREMENTS_PARTS; ++partIdx)
        {
            kSize segIndex = *(kSize*)kArrayList_At(obj->segIndices, partIdx);
            kBool valid = segIndex < count;
            GdkPartSegmentResultClass part;
            if (valid)
                part = parts[segIndex];
            else
                kTest(kMemZero(&part, sizeof(part)));

            // X
            if (measIdx < GdkToolCfg_MeasurementCount(config) &&
                GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, measIdx)))
                    TestGenericInput_OutputValueAndPoints(tool, measIdx, (k64f)part.center.x, valid, centerPoints, bBoxLines, areas, segIndex, output);
            measIdx++;
            // Y
            if (measIdx < GdkToolCfg_MeasurementCount(config) &&
                GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, measIdx)))
                    TestGenericInput_OutputValueAndPoints(tool, measIdx, (k64f)part.center.y, valid, centerPoints, bBoxLines, areas, segIndex, output);
            measIdx++;
            // W
            if (measIdx < GdkToolCfg_MeasurementCount(config) &&
                GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, measIdx)))
                    TestGenericInput_OutputValueAndPoints(tool, measIdx, part.width, valid, centerPoints, bBoxLines, areas, segIndex, output);
            measIdx++;
            // L
            if (measIdx < GdkToolCfg_MeasurementCount(config) &&
                GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, measIdx)))
                    TestGenericInput_OutputValueAndPoints(tool, measIdx, part.length, valid, centerPoints, bBoxLines, areas, segIndex, output);
            measIdx++;

            kPoint3d64f center64f = { part.center.x, part.center.y, part.heightMax };
            TestGenericInput_SendFeaturePoint(tool, output,
                TEST_GENERIC_INPUT_OUTPUT_FEATURE_START + partIdx, &center64f);
        }
    }
    kCatchEx(&e)
    {
        // Invalid measurements
        for (kSize measIdx = 0; measIdx < GdkToolCfg_MeasurementCount(config); measIdx++)
        {
            if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, measIdx)))
            {
                TestGenericInput_OutputValueAndPoints(tool, measIdx, 0, kFALSE, centerPoints, bBoxLines, areas, kSIZE_NULL, output);
            }
        }
        // Invalid Features
        kPoint3d64f errPoint = { k64F_NULL, k64F_NULL, k64F_NULL };
        for (kSize featIdx = 0; featIdx < GdkToolCfg_FeatureCount(config); featIdx++)
        {
            if (GdkFeatureCfg_Enabled(GdkToolCfg_FeatureAt(config, featIdx)))
            {
                TestGenericInput_SendFeaturePoint(tool, output,
                    TEST_GENERIC_INPUT_OUTPUT_FEATURE_START + featIdx, &errPoint);
            }
        }
        kEndCatchEx(kOK);
    }
    kFinallyEx
    {
        kEndFinallyEx();
    }

    return kOK;
}

kStatus TestGenericInput_OutputValueAndPoints(kObject tool, kSize index, k64f value, kBool valid,
                vector<kPoint3d32f> centerPointList, vector<kPoint3d32f> rectanglePointList, vector<k64f> partAreaList,
                kSize highlightIndex, GdkToolOutput output)
{
    kObj(TestGenericInput, tool);
    GdkGraphic graphic = kNULL;
    GdkGraphicLineSet lineSet = kNULL;
    GdkGraphicPointSet pointSet = kNULL;
    kStatus exception = kOK;
    kArrayList rectPts = kNULL;
    k64f* partAreas = partAreaList.data();
    GdkGraphicLabel label = kNULL;
    kPoint3d64f pt64;
    kAlloc alloc;

    alloc = GdkTool_MessageAlloc(tool);

    kTry
    {
        if (index != kSIZE_NULL)
        {
            kTest(GdkGraphic_Construct(&graphic, alloc));

            if (centerPointList.size() > 0)
            {
                kTest(GdkGraphicPointSet_Construct(&pointSet, 3.0, kMARKER_SHAPE_CROSS, kCOLOR_CYAN,
                    centerPointList.data(), centerPointList.size(), alloc));
                kTest(GdkGraphic_AddPointSet(graphic, pointSet));
                pointSet = kNULL;

                if (obj->showDetails && partAreaList.size() > 0)
                {
                    kChar str_index[30] = "";
                    for (kSize i = 0; i < centerPointList.size(); i++)
                    {
                        // Set a label with the details
                        kTest(kStrPrintf(str_index, sizeof(str_index), "Index: %d\nArea: %.2f", i + 1, partAreas[i]));
                        GdkGraphicLabel_Construct(&label, kNULL);
                        GdkGraphicLabel_SetText(label, str_index);

                        kPoint3d_Init_(&pt64, centerPointList[i].x, centerPointList[i].y, centerPointList[i].z);
                        GdkGraphicLabel_SetPosition(label, &pt64);
                        GdkGraphic_AddLabel(graphic, label);

                        label = kNULL;
                    }
                }
            }
            if (rectanglePointList.size() > 0)
            {
                kSize i, j;
                kPoint3d32f ptTemp;
                kTest(kArrayList_Construct(&rectPts, kTypeOf(kPoint3d32f), TEST_GENERIC_INPUT_POINT_COUNT_PER_PART, alloc));

                kSize count = centerPointList.size();
                for (i = 0; i < count; i++)
                {
                    kTest(kArrayList_Clear(rectPts));
                    for (j = 0; j < TEST_GENERIC_INPUT_POINT_COUNT_PER_PART; ++j)
                    {
                        ptTemp = rectanglePointList[(i * TEST_GENERIC_INPUT_POINT_COUNT_PER_PART) + j];
                        kTest(kArrayList_Add(rectPts, &ptTemp));
                    }

                    if (i == highlightIndex)
                    {
                        kTest(GdkGraphicLineSet_Construct(&lineSet, 3.0, kCOLOR_RED,
                            (kPoint3d32f*)kArrayList_Data(rectPts), TEST_GENERIC_INPUT_POINT_COUNT_PER_PART, alloc));
                    }
                    else
                    {
                        kTest(GdkGraphicLineSet_Construct(&lineSet, 3.0, kCOLOR_CYAN,
                            (kPoint3d32f*)kArrayList_Data(rectPts), TEST_GENERIC_INPUT_POINT_COUNT_PER_PART, alloc));
                    }
                    kTest(GdkGraphic_AddLineSet(graphic, lineSet));
                    lineSet = kNULL;
                }
            }

            kTest(GdkToolOutput_SetRendering(output, index, graphic));

            if (valid) {
                kTest(GdkToolOutput_SetResult(output, index, value, GDK_MEASUREMENT_OK));
            }
            else {
                kTest(GdkToolOutput_SetResult(output, index, k64F_NULL, GDK_MEASUREMENT_ERROR_VALUE));
            }
            graphic = kNULL;
        }
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kDestroyRef(&graphic);
        kDestroyRef(&lineSet);
        kDestroyRef(&pointSet);
        kDestroyRef(&rectPts);
        kDestroyRef(&label);

        kEndFinallyEx();
    }

    return kOK;
}

// Output the center as a point feature
kStatus TestGenericInput_SendFeaturePoint(TestGenericInput tool, GdkToolOutput output, kSize index, const kPoint3d64f* point)
{
    GdkPointFeature feature = kNULL;
    GdkToolCfg config = GdkTool_Config(tool);

    kSize count = GdkToolCfg_FeatureCount(config);
    if (index >= count)
    {
        kLogf("Error when setting feature point.");
        return kERROR;
    }

    feature = (GdkPointFeature)GdkToolOutput_FeatureAt(output, index);
    if (GdkFeature_Type(feature) != GDK_FEATURE_TYPE_POINT)
    {
        kLogf("Feature point is not a point.");
        return kERROR;
    }

    kCheck(GdkPointFeature_SetPosition(feature, point));
    return kOK;
}
