/*
* This class is used to collect one or more time stamps during an iteration,
* and to collect the iterations together.
*
* An iteration encompasses a set of actions that take place based on some trigger.
* The same set of actions are expected to take place when the next trigger occurs.
*
* One use is to collect time stamps in a processing path for some input data.
* Each pass through the processing path is an iteration. Within an iteration of the processing
* path, it may be useful to know the duration of each step in the path takes. The
* duration can be calculated by taking the difference between collector collected
* before and after the step.
*
* The output is a text file stored in /user directory.
* Each line contains one iteration's timestamps in the following format
* (where "ts" means "timestamp").
*
*      <iteration#>,<ts1>,<ts2>,...,<tsn>
*
* The file can be imported into a spreadsheet program to calculate time deltas.
*
* !!!! WARNING !!!!
* This class MUST NEVER be used in production code because the time stamp
* collection slows down product performance.
*
* EXAMPLE USAGE:
*   - example is based on using the GtDimensionTool.c.
*
* 1. Create the collector
*
*      GtFx(kStatus) GtDimensionTool_VInit(GtDimensionTool info, kType type, kAlloc alloc)
*      {
*          ....
*          // Create the collector to collect 2 timestamps.
*          GdkTimeStampCollector_Construct(&obj->collector, 2, alloc);
*          ....
*      }
*
* 2. Clear the timestamp list in preparation for collecting timestamps.
*
*      GtFx(kStatus) GtDimensionTool_VStart(GtDimensionTool tool)
*      {
*          ....
*          GdkTimeStampCollector_Start(obj->collector);
*          ....
*      }
*
* 3. Mark the start, collect two timestamp and then mark the stop of an iteration
*    to save the collected timestamps.
*
*      GtFx(kStatus) GtDimensionTool_VProcess(GtDimensionTool tool, GdkToolInput input, GdkToolOutput output)
*      {
*          ....
*          GdkTimeStampCollector_StartIteration(obj->collector);
*          ....
*          GdkTimeStampCollector_AddTimeStamp(obj->collector, 0);
*          ....
*          GdkTimeStampCollector_AddTimeStamp(obj->collector, 1);
*          ....
*          GdkTimeStampCollector_EndIteration(obj->collector);
*          ....
*      }
*
* 4. Stop collecting timestamps and output to a file.
*
*      GtFx(kStatus) GtDimensionTool_VStop(GtDimensionTool tool)
*      {
*          ....
*          GdkTimeStampCollector_Stop(obj->collector);
*          ....
*      }
*
* 5. Free the collector object
*
*      GtFx(kStatus) GtDimensionTool_VRelease(GtDimensionTool info)
*      {
*          ....
*          kDestroyRef(&obj->collector);
*          ....
*      }
*
*/
#ifndef GDK_TIMESTAMP_COLLECTOR_H
#define GDK_TIMESTAMP_COLLECTOR_H

#include <Gdk/GdkDef.h>

typedef kObject GdkTimeStampCollector;
kDeclareClassEx(Gdk, GdkTimeStampCollector, kObject)

GdkFx(kStatus) GdkTimeStampCollector_Construct(GdkTimeStampCollector* collector, k32u numStamps, kAlloc alloc);

GdkFx(k32s) GdkTimeStampCollector_Start(GdkTimeStampCollector collector);
GdkFx(kStatus) GdkTimeStampCollector_Stop(GdkTimeStampCollector collector);
GdkFx(k32s) GdkTimeStampCollector_StartIteration(GdkTimeStampCollector collector);
GdkFx(k32s) GdkTimeStampCollector_EndIteration(GdkTimeStampCollector collector);
GdkFx(kStatus) GdkTimeStampCollector_AddTimeStamp(GdkTimeStampCollector collector, kSize entryNum);

#endif  // GDK_TIMESTAMP_COLLECTOR_H
