/**
* @file    GdkToolInfo.h
* @brief   Declares the GdkToolInfo class.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_TOOL_INFO_H
#define GDK_TOOL_INFO_H

#include <Gdk/GdkDef.h>
#include <Gdk/Tools/GdkMeasurementInfo.h>
#include <Gdk/Tools/GdkFeatureInfo.h>
#include <Gdk/Tools/GdkDiagnosticInfo.h>
#include <Gdk/Tools/GdkToolDataOutputInfo.h>
#include <Gdk/Tools/GdkToolVersionInfo.h>
#include <Gdk/Config/GdkParamsInfo.h>

/**
* Describes a tool type which is essentially the class description
* of the tool to be used to instantiate instances of the tool.
*
* @class       GdkToolInfo
* @extends     kObject
* @ingroup     Gdk-Tools
*/
typedef kObject GdkToolInfo;
kDeclareClassEx(Gdk, GdkToolInfo, kObject)

/**
* Sets the tool type name.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    name        Type name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetTypeName(GdkToolInfo info, const kChar* name);

/**
* Returns the tool type name.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Type name.
*/
GdkFx(const kChar*) GdkToolInfo_TypeName(GdkToolInfo info);

/**
* Enable or disable auto versioning. When disabled, the version is set to "1" if not already set.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    enable      Enable or disable.
* @return               Params info.
*/
GdkFx(kStatus) GdkToolInfo_EnableAutoVersion(GdkToolInfo info, kBool enable);

/**
* Sets the version name explicitly.
*
* When using multiple versions, this sets the name of the version at index 0.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    name        Version name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetVersionName(GdkToolInfo info, const kChar* name);

/**
* Sets the tool display label.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    label       Display label.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetLabel(GdkToolInfo info, const kChar* label);

/**
* Returns the tool display label.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Display label.
*/
GdkFx(const kChar*) GdkToolInfo_Label(GdkToolInfo info);

/**
* Adds the data source type accepted by the tool.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    type        Data source type.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddSourceType(GdkToolInfo info, GdkDataType type);

/**
* Add an option to the list of acceptable data sources.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    source      Data source option.
* @return               Data type.
*/
GdkFx(kStatus) GdkToolInfo_AddSourceOption(GdkToolInfo info, GdkDataSource source);

/**
* Enables or disables anchoring capability for a specified anchoring parameter.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    axis        Anchoring parameter - axis or angle.
* @param    enabled     Enabled state.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_EnableAnchoring(GdkToolInfo info, GdkAnchorParam axis, kBool enabled);

/**
* Checks anchoring capability for a specified anchoring parameter.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    axis        Anchoring parameter - axis or angle.
* @return               Whether or not anchoring capability is enabled.
*/
GdkFx(kBool) GdkToolInfo_AnchoringEnabled(GdkToolInfo info, GdkAnchorParam axis);

/**
* Returns the schema for the parameters.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Params info.
*/
GdkFx(GdkParamsInfo) GdkToolInfo_Params(GdkToolInfo info);

/**
* Marks the tool as requiring a tool-license
*
* @private                  @memberof GdkToolInfo
* @param    info            info object
* @param    requiresLicense Boolean: Does the tool require a license?
* @return                   Status of operation
*/
GdkFx(kStatus) GdkToolInfo_SetRequiresLicense(GdkToolInfo info, kBool requiresLicense);

/**
* Returns true if the tool requires a tool license
*
* @private                  @memberof GdkToolInfo
* @param    info            info object
* @return                   kTRUE if tool requires a license
*/
GdkFx(kBool) GdkToolInfo_RequiresLicense(GdkToolInfo info);

/**
* Adds a data output type to a tool with the specified data type

* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    dataType        Tool data output type of type GdkDataType. The following types are supported:
*                           - #GDK_DATA_TYPE_MEASUREMENT
*                           - #GDK_DATA_TYPE_FEATURE_POINT
*                           - #GDK_DATA_TYPE_FEATURE_LINE
*                           - #GDK_DATA_TYPE_FEATURE_CIRCLE
*                           - #GDK_DATA_TYPE_FEATURE_PLANE
*                           - #GDK_DATA_TYPE_UNIFORM_PROFILE
*                           - #GDK_DATA_TYPE_UNIFORM_SURFACE
*                           - #GDK_DATA_TYPE_SURFACE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_PROFILE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_MESH
*                           - #GDK_DATA_TYPE_GENERIC_BASE to N
* @param    name            Name of the output type.
* @param    label           Label of the output.
* @param    outputInfo      Pointer to receive the new data output's info object.
*                           It is based on the dataType and should be one of the following:
*                              GdkMeasurementInfo
*                              GdkFeatureInfo
*                              GdkToolDataOutputInfo
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddOutput(GdkToolInfo info, GdkDataType dataType, const kChar* name, const kChar* label, GdkOutputInfo *outputInfo);

/**
* Adds an input data set to a tool with the specified data type

* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    dataType        Tool data output type of type GdkDataType. The following types are supported:
*                           - #GDK_DATA_TYPE_FEATURE_POINT
*                           - #GDK_DATA_TYPE_FEATURE_LINE
*                           - #GDK_DATA_TYPE_FEATURE_CIRCLE
*                           - #GDK_DATA_TYPE_FEATURE_PLANE
*                           - #GDK_DATA_TYPE_UNIFORM_PROFILE
*                           - #GDK_DATA_TYPE_PROFILE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_UNIFORM_SURFACE
*                           - #GDK_DATA_TYPE_SURFACE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_MESH
*                           - #GDK_DATA_TYPE_GENERIC_BASE to N
* @param    name            Name of the input type.
* @param    label           Label of the input. If kNULL, will be set to same as name parameter
* @param    paramInfo       Pointer to receive the new input data set's param info object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddInput(GdkToolInfo info, GdkDataType dataType, const kChar* name, const kChar* label, GdkParamInfo *paramInfo);

/**
* Adds a parameter to a tool

* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    paramType       Tool parameter type. The following types are supported:
*                           - #GDK_PARAM_TYPE_INT
*                           - #GDK_PARAM_TYPE_FLOAT
*                           - #GDK_PARAM_TYPE_BOOL
*                           - #GDK_PARAM_TYPE_STRING
*                           - #GDK_PARAM_TYPE_PROFILE_REGION
*                           - #GDK_PARAM_TYPE_SURFACE_REGION
*                           - #GDK_PARAM_TYPE_SURFACE_REGION_2D
*                           - #GDK_PARAM_TYPE_POINT_SET_REGION
* @param    name            Name of the parameter.
* @param    label           Label of the param. If kNULL, will be set to same as name parameter
* @param    defValue        Pointer to the default value to set the parameter. This depends on the paramType passed.
* @param    paramInfo       Pointer to receive the new input data set's paramInfo object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddParam(GdkToolInfo info, GdkParamType paramType, const kChar* name, const kChar* label, const void* defValue, GdkParamInfo *paramInfo);

/**
* Adds a tool configuration version.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    name        Version name (can be empty string).
* @param    newVersion  Pointer to receive new version object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddVersion(GdkToolInfo info, const kChar* name, GdkToolVersionInfo* newVersion);

/**
* Returns the number of version objects. Note that this always at least 1.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Number of version objects.
*/
GdkFx(kSize) GdkToolInfo_VersionCount(GdkToolInfo info);

/**
* Returns the version object at the specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Index of version object.
* @return               Version object at index.
*/
GdkFx(GdkToolVersionInfo) GdkToolInfo_VersionAt(GdkToolInfo info, kSize index);

/**
* Return the default version object. This is the version information that is
* used when a new tool is instantiated.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Version object.
*/
GdkFx(GdkToolVersionInfo) GdkToolInfo_DefaultVersion(GdkToolInfo info);

/**
* Sets the default version object.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    version     Version object that should be used as the default for new tool instances.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetDefaultVersion(GdkToolInfo info, GdkToolVersionInfo version);

GdkFx(GdkToolFormat) GdkToolInfo_Format(GdkToolInfo info);


//// Deprecated API functions

/**
* [Deprecated] Use GdkToolInfo_AddSourceType() instead.
*
* Sets the data source type accepted by the tool.
*
* @deprecated
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    type        Data source type.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetSourceType(GdkToolInfo info, GdkDataType type);

/**
* [Deprecated] Use GdkToolInfo_AddOutput() instead.
*
* Add a measurement type.
*
* @deprecated
* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    type            Measurement type name (should be unique within the tool).
* @param    measurementInfo Pointer to receive the new measurement info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddMeasurement(GdkToolInfo info, const kChar* type, GdkMeasurementInfo* measurementInfo);

/**
* [Deprecated] Use GdkTool_MeasurementIndexByType() to retrieve index of outputs instead.
*
* Returns the number of measurement types.
*
* @deprecated
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Measurement type count.
*/
GdkFx(kSize) GdkToolInfo_MeasurementCount(GdkToolInfo info);

/**
* [Deprecated] Use GdkToolInfo_AddOutput() instead.
*
* Add a feature type.
*
* @deprecated
* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    type            Feature Type String
* @param    dataType        Feature data type (Point, Circle, etc).
* @param    featureInfo Pointer to receive the new feature info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddFeature(GdkToolInfo info, const kChar* type, GdkFeatureDataType dataType, GdkFeatureInfo* featureInfo);

/**
* [Deprecated] Use GdkTool_FeatureIndexByType() to retrieve index of outputs instead.
*
* Returns the number of feature types.
*
* @deprecated
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Feature type count.
*/
GdkFx(kSize) GdkToolInfo_FeatureCount(GdkToolInfo info);

/**
* [Deprecated] Use GdkToolInfo_AddOutput() instead.
*
* Adds a tool data output to a tool with the specified index
*
* @deprecated
* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    name            Tool data output name.
* @param    dataType        Tool data type of the output.
* @param    tdOutputInfo    Pointer to receive the new tool data output's info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddToolDataOutput(GdkToolInfo info, const kChar* name, GdkDataType dataType, GdkToolDataOutputInfo* tdOutputInfo);

/**
* [Deprecated] Use GdkTool_DataOutputIndexByType() to retrieve index of outputs instead.
*
* Returns the number of tool data outputs.
*
* @deprecated
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Tool data output count.
*/
GdkFx(kSize) GdkToolInfo_ToolDataOutputCount(GdkToolInfo info);


#include <Gdk/Tools/GdkToolInfo.x.h>

#endif 
