/**
* @file    GdkToolCfg.h
* @brief   Declares the GdkToolCfg class.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_TOOL_CFG_H
#define GDK_TOOL_CFG_H

#include <Gdk/Config/GdkMeasurementCfg.h>
#include <Gdk/Config/GdkFeatureCfg.h>
#include <Gdk/Config/GdkToolDataOutputCfg.h>
#include <Gdk/Config/GdkParams.h>

#include <Gdk/GdkDef.h>

/**
* Represents the configuration instance for a tool.
* Where the GdkToolInfo is used to define the tool class, the GdkToolCfg
* is used to interface with the instance of the corresponding tool class.
*
* @class       GdkToolCfg
* @ingroup     Gdk-Config
*/
typedef kPointer GdkToolCfg;

/**
* Returns the tool name.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Pointer to type name.
*/
GdkFx(const kChar*) GdkToolCfg_Name(GdkToolCfg config);

/**
* Set the data source used by the tool.
*
* This function should be called only during configuration initialization.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    source      Data source.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolCfg_SetSource(GdkToolCfg config, GdkDataSource source);

/**
* Returns the data source used by the tool.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Data source.
*/
GdkFx(GdkDataSource) GdkToolCfg_Source(GdkToolCfg config);

/**
* Set the anchor source used by the tool.
*
* This function should be called only during configuration initialization.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    axis        Anchoring parameter - axis or angle.
* @param    id          ID of source measurement.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolCfg_SetAnchor(GdkToolCfg config, GdkAnchorParam axis, k32s id);

/**
* Returns the anchor source used by the tool.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    axis        Anchoring parameter - axis or angle.
* @return               ID of source measurement.
*/
GdkFx(k32s) GdkToolCfg_Anchor(GdkToolCfg config, GdkAnchorParam axis);

/**
* Returns user-defined parameter set.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Parameter set.
*/
GdkFx(GdkParams) GdkToolCfg_Parameters(GdkToolCfg config);

/**
* Add a new measurement to the tool.
*
* @public                   @memberof GdkToolCfg
* @param    config          Configuration object.
* @param    type            Type name of the measurement.
* @param    measurementCfg  Pointer to receive the new measurement.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolCfg_AddMeasurement(GdkToolCfg config, const kChar* type, GdkMeasurementCfg* measurementCfg);

/**
* Returns the number of measurements in the tool.
*
* @public               @memberof GdkToolCfg
* @return               Number of measurements.
*/
GdkFx(kSize) GdkToolCfg_MeasurementCount(GdkToolCfg config);

/**
* Returns the measurement at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of measurement.
* @return               Measurement handle.
*/
GdkFx(GdkMeasurementCfg) GdkToolCfg_MeasurementAt(GdkToolCfg config, kSize index);

/**
* Removes the measurement at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of measurement.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolCfg_RemoveMeasurement(GdkToolCfg config, kSize index);

/**
* Add a new feature to the tool.
*
* @public                   @memberof GdkToolCfg
* @param    config          Configuration object.
* @param    type            Type of the feature.
* @param    featureCfg      Pointer to receive the new feature.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolCfg_AddFeature(GdkToolCfg config, const kChar* type, GdkFeatureCfg* featureCfg);

/**
* Returns the number of features in the tool.
*
* @public               @memberof GdkToolCfg
* @return               Number of features.
*/
GdkFx(kSize) GdkToolCfg_FeatureCount(GdkToolCfg config);

/**
* Returns the feature at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of feature.
* @return               Feature handle.
*/
GdkFx(GdkFeatureCfg) GdkToolCfg_FeatureAt(GdkToolCfg config, kSize index);

/**
* Removes the feature at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of feature.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolCfg_RemoveFeature(GdkToolCfg config, kSize index);

/**
* Add a new data output to the tool.
*
* @public                   @memberof GdkToolCfg
* @param    config          Configuration object.
* @param    type            Type of the data output.
* @param    dataOutputCfg   Pointer to receive the new data outpout.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolCfg_AddToolDataOutput(GdkToolCfg config, const kChar* type, GdkToolDataOutputCfg* dataOutputCfg);

/**
* Returns the number of data outputs in the tool.
*
* @public               @memberof GdkToolCfg
* @return               Number of data outputs.
*/
GdkFx(kSize) GdkToolCfg_ToolDataOutputCount(GdkToolCfg config);

/**
* Returns the data output at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of data output.
* @return               Tool data handle.
*/
GdkFx(GdkToolDataOutputCfg) GdkToolCfg_ToolDataOutputAt(GdkToolCfg config, kSize index);

/**
* Removes the tool data output at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of output.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolCfg_RemoveToolDataOutput(GdkToolCfg config, kSize index);

/**
* Returns the version of the configuration.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Config version.
*/
GdkFx(const kChar*) GdkToolCfg_Version(GdkToolCfg config);

/**
* Returns whether the version not set.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Flag indicating whether the version is set.
*/
GdkFx(kBool) GdkToolCfg_VersionUnknown(GdkToolCfg config);

/**
* Returns the parameter at an index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of parameter.
* @return               Parameter handle.
*/
GdkFx(GdkParam) GdkToolCfg_ParameterAt(GdkToolCfg config, kSize index);

/**
* Returns the parameter matching the name passed.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    name        Name of parameter.
* @return               Matching parameter handle or kNULL if not found
*/
GdkFx(GdkParam) GdkToolCfg_FindParameter(GdkToolCfg config, const kChar *name);

/**
* Returns the count of parameters in the tool.
*
* @public               @memberof GdkToolCfg
* @return               Number of parameters.
*/
GdkFx(kSize) GdkToolCfg_ParameterCount(GdkToolCfg config);

/**
* Returns the count of outputs in the tool.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @return               Number of outputs currently instantiated in the tool.
*/
GdkFx(kSize) GdkToolCfg_OutputCount(GdkToolCfg config);

/**
* Returns the output of the tool at the index.
*
* @public               @memberof GdkToolCfg
* @param    config      Configuration object.
* @param    index       Index of all the outputs (unified index).
* @return               Output data handle corresponding to the index which could be one of 
*                       GdkMeasurementCfg, GdkFeatureCfg or GdkToolDataOutputCfg.
*/
GdkFx(GdkToolOutputCfg) GdkToolCfg_OutputAt(GdkToolCfg config, kSize index);

#endif 
