#ifndef GS_CFG_EXT_PARAMS_H
#define GS_CFG_EXT_PARAMS_H

#include <Gdk/GdkDef.h>
#include <Gdk/Config/GdkParamInfo.h>
#include <GoApi/Configuration/Nodes/Container.h>
#include <GoApi/Configuration/Nodes/DynamicContainer.h>
#include <GoApi/Configuration/Attributes.h>
#include <GoApi/Configuration/Nodes/Value.h>
#include <GoApi/GoApi.h>

class ExtParamLUT;

typedef struct ExtParamsStatic
{
    ExtParamLUT* extParamsLut;
} ExtParamsStatic;

kDeclareStaticClassEx(Gdk, ExtParams)

GdkFx(kStatus) xExtParams_InitStatic();
GdkFx(kStatus) xExtParams_ReleaseStatic();

namespace GoCfg = Go::Configuration;

// Dedicated struct to enable custom serialization.
struct ExtParamType
{
    GdkParamType type;

    ExtParamType() : type(GDK_PARAM_TYPE_UNKNOWN)
    {
    }

    ExtParamType(GdkParamType type) : type(type)
    {
    }

    operator GdkParamType() const
    {
        return type;
    }

    ExtParamType& operator=(GdkParamType type)
    {
        this->type = type;
        return *this;
    }
};

///@cond IGNORE
GdkCppFx(std::ostream&) operator<<(std::ostream& stream, const ExtParamType& source);
GdkCppFx(std::istream&) operator>>(std::istream& stream, ExtParamType& source);
///@endcond

/**
* @class    ExtParam
* @ingroup  Gdk-Config
* @brief    Shared interface of all GDK parameters.
*           Note all parameters derive from both this and a model node type.
*/
struct ExtParam
{
    using UsedAttrType = GoCfg::UsedAttribute;
    using StringAttrType = GoCfg::ValueAttribute<std::string>;
    using ParamTypeAttrType = GoCfg::ValueAttribute<ExtParamType>;
    using StringOptionsType = GoCfg::OptionsAttribute<std::string>;

    virtual ~ExtParam() {}

    virtual void Dispose() = 0;

    virtual GoCfg::Node& ModelNode() = 0;

    virtual UsedAttrType& Used() = 0;
    virtual ParamTypeAttrType& ParamType() = 0;
    virtual StringAttrType& Label() = 0;
    virtual StringAttrType& Units() = 0;

    virtual StringOptionsType& OptionNames() = 0;
    virtual void ClearOptions() = 0;
    virtual size_t OptionCount() const = 0;
    virtual void RemoveOptionAt(size_t index) = 0;

    virtual bool IsRead() const = 0;

    virtual GdkParamInfo ParamInfo() = 0;
    virtual GdkParamInfo EditableParamInfo() = 0;
    virtual void SetParamInfo(GdkParamInfo paramInfo) = 0;

    virtual void CopyValues(const ExtParam& other) = 0;
};

/**
* @class    ExtParamImpl
* @extends  ExtParam
* @ingroup  Gdk-Config
* @brief    Shared common implementation of parameters. Extends a chosen model node
*           type as template parameter "T".
*/
template <class T>
struct ExtParamImpl : public T, public ExtParam
{
    ParamTypeAttrType type;
    StringAttrType label;
    StringAttrType units;
    StringOptionsType optionNames;
    UsedAttrType used;

    GdkParamInfo paramInfo;
    GdkParamInfo editableParamInfo;

    ExtParamImpl()
    {
        this->RegisterAttribute(used);
        this->used = true;
        this->RegisterAttribute(type, "type");
        this->RegisterAttribute(label, "label");
        this->RegisterAttribute(units, "units");
        this->RegisterAttribute(optionNames, "optionNames");

        // Enabled by relevant derived classes (move into derived?).
        optionNames.enabled = false;

        paramInfo = kNULL;
        editableParamInfo = kNULL;

        // Perhaps these should just be virtual methods to begin with?
        T::SetPreWriteHandler([=] () {
            this->OnBeforeWrite();
        });

        T::SetPostReadHandler([=] () {
            this->OnAfterRead();
        });
    }

    ~ExtParamImpl()
    {
        kDestroyRef(&editableParamInfo);
    }

    void Dispose()
    {
        delete this;
    }

    void CopyValues(const ExtParam& other) override
    {
        auto& typedOther = dynamic_cast<const ExtParamImpl<T>&>(other);

        // GdkParamInfo fields are deliberately skiped - they contain meta data
        // related to the type of the parameter, which doesn't have to be and
        // shouldn't be updated.
        type = typedOther.type.Get();
        label = typedOther.label.Get();
        units = typedOther.units.Get();
        optionNames.CopyOptions(typedOther.optionNames);
    }

    GoCfg::Node& ModelNode() override
    {
        return static_cast<GoCfg::Node&>(*this);
    }

    ParamTypeAttrType& ParamType() override
    {
        return type;
    }

    UsedAttrType& Used() override
    {
        return used;
    }

    StringAttrType& Label() override
    {
        return label;
    }

    StringAttrType& Units() override
    {
        return units;
    }

    StringOptionsType& OptionNames() override
    {
        return optionNames;
    }

    void ClearOptions() override
    {
        optionNames.Clear();
    }

    size_t OptionCount() const override
    {
        return optionNames.Count();
    }

    void RemoveOptionAt(size_t index)
    {
        optionNames.Remove(index);
    }

    bool IsRead() const override
    {
        return T::IsRead();
    }

    GdkParamInfo ParamInfo() override
    {
        return !kIsNull(editableParamInfo) ? editableParamInfo : paramInfo;
    }

    GdkParamInfo EditableParamInfo() override
    {
        if (kIsNull(editableParamInfo) && !kIsNull(paramInfo))
        {
            Go::Test(GdkParamInfo_ConstructParamAttached(&editableParamInfo, (GdkParam)static_cast<ExtParam*>(this), kNULL));
        }

        return editableParamInfo;
    }

    void SetParamInfo(GdkParamInfo paramInfo) override
    {
        this->paramInfo = paramInfo;
        kDestroyRef(&editableParamInfo);
    }

protected:
    virtual void OnBeforeWrite()
    {
        units.enabled = (units.Get() != "");
        used.enabled = (used.Get() != true);
    };

    virtual void OnAfterRead()
    {
    }
};

/**
* @class    ExtValueParam
* @extends  ExtParamImpl
* @ingroup  Gdk-Config
* @brief    Base simple value parameter.
*/
template <class T>
struct ExtValueParam : ExtParamImpl<GoCfg::Value<T>>
{
    using BaseType = ExtParamImpl<GoCfg::Value<T>>;
    GoCfg::OptionsAttribute<T> options;

    ExtValueParam()
    {
        GoCfg::Value<T>::RegisterAttribute(options, "options");
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtParamImpl<GoCfg::Value<T>>::CopyValues(other); // Copy base

        auto& typedOther = dynamic_cast<const ExtValueParam<T>&>(other);

        this->Set(typedOther.Get()); // Copy the actual value
        options.CopyOptions(typedOther.options);
    }

    void ClearOptions() override
    {
        options.Clear();
        this->optionNames.Clear();
    }

    size_t OptionCount() const override
    {
        return options.Count();
    }

    void RemoveOptionAt(size_t index)
    {
        options.Remove(index);

        if (index < this->optionNames.Count())
        {
            this->optionNames.Remove(index);
        }
    }

    using GoCfg::Value<T>::operator=;

protected:
    void OnBeforeWrite() override
    {
        BaseType::OnBeforeWrite();

        bool enableOptions = options.Count() > 0;

        options.enabled = enableOptions;
        this->optionNames.enabled = enableOptions;
    }
};

template <class T>
struct ExtNumericParam : ExtValueParam<T>
{
    GoCfg::MinAttribute<T> min;
    GoCfg::MaxAttribute<T> max;

    ExtNumericParam()
    {
        GoCfg::Value<T>::RegisterAttribute(min, "min");
        GoCfg::Value<T>::RegisterAttribute(max, "max");
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtValueParam<T>::CopyValues(other); // Copy base

        auto& typedOther = dynamic_cast<const ExtNumericParam<T>&>(other);
        min = typedOther.min.Get();
        max = typedOther.max.Get();
    }

    using ExtValueParam<T>::operator=;

protected:
    void OnBeforeWrite() override
    {
        ExtValueParam<T>::OnBeforeWrite();
    }
};

/**
* @class    ExtCompositeParam
* @extends  ExtParamImpl
* @ingroup  Gdk-Config
* @brief    Base composite value parameter that contain child properties.
*/
struct ExtCompositeParam : ExtParamImpl<GoCfg::Container>
{
    ExtCompositeParam()
    {
    }
};

// GDK_PARAM_TYPE_INT
struct ExtIntParam : ExtNumericParam<k32s>
{
    ExtIntParam()
    {
        optionNames.enabled = true;

        min = k32S_NULL;
        max = k32S_NULL;
    }

    using ExtNumericParam<k32s>::operator=;

protected:
    void OnBeforeWrite() override
    {
        ExtNumericParam<k32s>::OnBeforeWrite();

        min.enabled = min.Get() != k32S_NULL;
        max.enabled = max.Get() != k32S_NULL;
    }
};

// GDK_PARAM_TYPE_FLOAT
struct ExtFloatParam : ExtNumericParam<k64f>
{
    ExtFloatParam()
    {
        optionNames.enabled = true;

        min = k64F_NULL;
        max = k64F_NULL;
    }

    using ExtNumericParam<k64f>::operator=;

protected:
    void OnBeforeWrite() override
    {
        ExtNumericParam<k64f>::OnBeforeWrite();

        min.enabled = min.Get() != k64F_NULL;
        max.enabled = max.Get() != k64F_NULL;
    }
};

// GDK_PARAM_TYPE_BOOL
struct ExtBoolParam : ExtValueParam<bool>
{
    ExtBoolParam()
    {
    }

    using ExtValueParam<bool>::operator=;
};

// GDK_PARAM_TYPE_STRING
typedef ExtValueParam<std::string> ExtStringParam;

struct ExtRegionParam : ExtCompositeParam
{
    ExtRegionParam()
    {
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtCompositeParam::CopyValues(other); // Copy base
    }
};

// GDK_PARAM_TYPE_PROFILE_REGION
struct ExtProfileRegionParam : ExtRegionParam
{
    GoCfg::Value<k64f> x;
    GoCfg::Value<k64f> z;
    GoCfg::Value<k64f> width;
    GoCfg::Value<k64f> height;

    ExtProfileRegionParam()
    {
        Register(x, "X");
        Register(z, "Z");
        Register(width, "Width");
        Register(height, "Height");
    }

    GdkRegionXZ64f GdkRegion() const
    {
        GdkRegionXZ64f gdkRegion;

        gdkRegion.x = x;
        gdkRegion.z = z;
        gdkRegion.width = width;
        gdkRegion.height = height;

        return gdkRegion;
    }

    const GdkRegionXZ64f* GdkRegionPtr()
    {
        gdkRegionStore = GdkRegion();
        return &gdkRegionStore;
    }

    void SetGdkRegion(const GdkRegionXZ64f& region)
    {
        x = region.x;
        z = region.z;
        width = region.width;
        height = region.height;
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtRegionParam::CopyValues(other); // Copy base

        auto& typedOther = dynamic_cast<const ExtProfileRegionParam&>(other);
        SetGdkRegion(typedOther.GdkRegion());
    }

private:
    GdkRegionXZ64f gdkRegionStore;
};

// GDK_PARAM_TYPE_SURFACE_REGION
struct ExtSurfaceRegionParam : ExtRegionParam
{
    GoCfg::Value<k64f> x;
    GoCfg::Value<k64f> y;
    GoCfg::Value<k64f> z;
    GoCfg::Value<k64f> width;
    GoCfg::Value<k64f> length;
    GoCfg::Value<k64f> height;
    GoCfg::Value<k64f> zAngle;
    GoCfg::UsedAttribute zAngleUsed;

    ExtSurfaceRegionParam()
    {
        Register(x, "X");
        Register(y, "Y");
        Register(z, "Z");
        Register(width, "Width");
        Register(length, "Length");
        Register(height, "Height");
        Register(zAngle, "ZAngle");
        zAngle.RegisterAttribute(zAngleUsed);
    }

    GdkRegion3d64f GdkRegion() const
    {
        GdkRegion3d64f gdkRegion;

        gdkRegion.x = x;
        gdkRegion.y = y;
        gdkRegion.z = z;
        gdkRegion.width = width;
        gdkRegion.length = length;
        gdkRegion.height = height;
        gdkRegion.zAngle = zAngle;

        return gdkRegion;
    }

    const GdkRegion3d64f* GdkRegionPtr()
    {
        gdkRegionStore = GdkRegion();
        return &gdkRegionStore;
    }

    void SetGdkRegion(const GdkRegion3d64f& region)
    {
        x = region.x;
        y = region.y;
        z = region.z;
        width = region.width;
        length = region.length;
        height = region.height;
        zAngle = region.zAngle;
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtRegionParam::CopyValues(other); // Copy base

        auto& typedOther = dynamic_cast<const ExtSurfaceRegionParam&>(other);
        SetGdkRegion(typedOther.GdkRegion());
        zAngleUsed = typedOther.zAngleUsed.Get(); // Set the z angle used flag
    }

private:
    GdkRegion3d64f gdkRegionStore;
};

// GDK_PARAM_TYPE_SURFACE_REGION_2D
struct ExtSurfaceRegion2dParam : ExtRegionParam
{
    GoCfg::Value<k64f> x;
    GoCfg::Value<k64f> y;
    GoCfg::Value<k64f> width;
    GoCfg::Value<k64f> length;

    ExtSurfaceRegion2dParam()
    {
        Register(x, "X");
        Register(y, "Y");
        Register(width, "Width");
        Register(length, "Length");
    }

    GdkRegion2d64f GdkRegion() const
    {
        GdkRegion2d64f gdkRegion;

        gdkRegion.x = x;
        gdkRegion.y = y;
        gdkRegion.width = width;
        gdkRegion.length = length;

        return gdkRegion;
    }

    const GdkRegion2d64f* GdkRegionPtr()
    {
        gdkRegionStore = GdkRegion();
        return &gdkRegionStore;
    }

    void SetGdkRegion(const GdkRegion2d64f& region)
    {
        x = region.x;
        y = region.y;
        width = region.width;
        length = region.length;
    }

    void CopyValues(const ExtParam& other) override
    {
        ExtRegionParam::CopyValues(other); // Copy base

        auto& typedOther = dynamic_cast<const ExtSurfaceRegion2dParam&>(other);
        SetGdkRegion(typedOther.GdkRegion());
    }

private:
    GdkRegion2d64f gdkRegionStore;
};

// GDK_PARAM_TYPE_GEOMETRIC_FEATURE
struct ExtFeatureInputParam : ExtValueParam<k32s>
{
    ExtFeatureInputParam()
    {
    }

protected:
    void OnBeforeWrite() override
    {
        ExtValueParam<k32s>::OnBeforeWrite();

        this->options.enabled = true;
        this->optionNames.enabled = true;
    }
};

struct PointSetRegionAxisValue : GoCfg::Value<k64f>
{
    GoCfg::ValueAttribute<k64f> actualValue;
    GoCfg::ValueAttribute<bool> readonly;

    PointSetRegionAxisValue()
    {
        this->RegisterAttribute(actualValue, "value");
        this->RegisterAttribute(readonly, "readonly");
    }

    using GoCfg::Value<k64f>::operator=;
};

struct PointSetRegionPoint : GoCfg::Container
{
    PointSetRegionAxisValue x;
    PointSetRegionAxisValue y;
    PointSetRegionAxisValue z;

    PointSetRegionPoint()
    {
        this->Register(x, "X");
        this->Register(y, "Y");
        this->Register(z, "Z");
    }

    const kPoint3d64f* Point3d64fPtr()
    {
        compatContainer.x = x;
        compatContainer.y = y;
        compatContainer.z = z;

        return &compatContainer;
    }

private:
    kPoint3d64f compatContainer;
};

// GDK_PARAM_TYPE_POINT_SET_REGION
struct ExtPointSetRegionParam : ExtParamImpl<GoCfg::DynamicContainer<PointSetRegionPoint>>
{
    using BaseType = ExtParamImpl<GoCfg::DynamicContainer<PointSetRegionPoint>>;

    GoCfg::ValueAttribute<k32u> maxPoints = k32U_MAX;
    GoCfg::ValueAttribute<k32u> minPoints = 0;
    GoCfg::ValueAttribute<kMarkerShape> pointShape = -1;
    GoCfg::ValueAttribute<k16u> pointSize = 0;
    GoCfg::ValueAttribute<GdkPointSetRegionMode> mode = -1;
    GoCfg::ValueAttribute<GdkPointSetRegionColor> pointColor = k32U_MAX;
    GoCfg::ValueAttribute<GdkPointSetRegionColor> lineColor = k32U_MAX;
    GoCfg::ValueAttribute<bool> showProjection = true;

    bool useShowProjection = true;
    k64f constantZ = k64F_NULL;

    ExtPointSetRegionParam()
    {
        // TODO: a lot of these attributes are optional
        this->RegisterAttribute(maxPoints, "maxPointCount");
        this->RegisterAttribute(minPoints, "minPointCount");
        this->RegisterAttribute(pointShape, "pointShape");
        this->RegisterAttribute(pointSize, "pointSize");
        this->RegisterAttribute(mode, "mode");
        this->RegisterAttribute(pointColor, "pointColor");
        this->RegisterAttribute(lineColor, "lineColor");
        this->RegisterAttribute(showProjection, "showProjection");

        this->RegisterType<PointSetRegionPoint>("Point");
    }

    void CopyValues(const ExtParam& other) override
    {
        BaseType::CopyValues(other);

        auto& typedOther = dynamic_cast<const ExtPointSetRegionParam&>(other);

        maxPoints = typedOther.maxPoints.Get();
        minPoints = typedOther.minPoints.Get();
        pointShape = typedOther.pointShape.Get();
        pointSize = typedOther.pointSize.Get();
        mode = typedOther.mode.Get();
        pointColor = typedOther.pointColor.Get();
        lineColor = typedOther.lineColor.Get();
        showProjection = typedOther.showProjection.Get();

        useShowProjection = typedOther.useShowProjection;
        constantZ = typedOther.constantZ;
    }

protected:
    void OnBeforeWrite() override
    {
        BaseType::OnBeforeWrite();

        mode.enabled = mode.Get() != -1;
        pointShape.enabled = pointShape.Get() != -1;
        pointSize.enabled = pointSize.Get() != 0;
        pointColor.enabled = pointColor.Get() != k32U_MAX;
        lineColor.enabled = lineColor.Get() != k32U_MAX;
        showProjection.enabled = useShowProjection;
        maxPoints.enabled = maxPoints.Get() != k32U_MAX;
        minPoints.enabled = minPoints.Get() != 0;

        for (size_t i = 0; i < this->Count(); ++i)
        {
            PointSetRegionPoint& point = this->At(i);

            point.x.actualValue = point.x.Get();
            point.x.readonly = false;

            point.y.actualValue = point.y.Get();
            point.y.readonly = false;

            if (constantZ == k64F_NULL)
            {
                point.z.actualValue = point.z.Get();
                point.z.readonly = false;
            }
            else
            {
                point.z.actualValue = constantZ;
                point.z.readonly = true;
            }
        }
    }
};

/**
* @class    ExtStreamId
* @ingroup  Gdk-Config
* @brief    Represents a stream ID. Supports usage as a model value.
*/
struct GdkClass ExtStreamId
{
    k32s step;
    k32s id;
    k32s source;

    ExtStreamId();
    ExtStreamId(const GdkStreamId& other);

    bool operator==(const ExtStreamId& other) const;
    bool operator!=(const ExtStreamId& other) const;

    operator GdkStreamId() const;
};

///@cond IGNORE
GdkCppFx(std::ostream&) operator<<(std::ostream& stream, const ExtStreamId& source);
GdkCppFx(std::istream&) operator>>(std::istream& stream, ExtStreamId& source);
///@endcond

// GDK_PARAM_TYPE_DATA_INPUT
struct ExtDataInputParam : ExtValueParam<ExtStreamId>
{
    // GOC-13328: store parameter data input's supported data types
    // defined for the tool's parameter so that the configuration
    // XML will show the data types as options for that parameter.
    GoCfg::OptionsAttribute<GdkDataType> dataTypes;

    ExtDataInputParam()
    {
        // GOC-13328: output the supported data types as a list of
        // option values.
        this->RegisterAttribute(dataTypes, "dataTypes");
    }

    // This clears this class's "dataTypes" plus the parent classes'
    // options: the ExtValueParam's "options" and ExtParamImpl's "optionNames".
    void ClearOptions() override
    {
        dataTypes.Clear();
        this->options.Clear();
        this->optionNames.Clear();
    }

protected:
    void OnBeforeWrite() override
    {
        ExtValueParam<ExtStreamId>::OnBeforeWrite();

        this->options.enabled = true;
        this->optionNames.enabled = true;
    }
};

/**
* @class    ExtParamSet
* @extends  DynamicContainer
* @ingroup  Gdk-Config
* @brief    Represents a set of properties.
*/
struct ExtParamSet : GoCfg::Container
{
    using ParamType = std::unique_ptr<ExtParam>;

    ExtParamSet()
    {
    }

    ExtParam& ParamAt(size_t index)
    {
        // ExtParam and Node are not directly related. Must dynamic cast.
        return dynamic_cast<ExtParam&>(At(index));
    }

    ExtParam& Find(const std::string& name)
    {
        size_t index = paramNameMap.at(name);
        return *(parameters[index].get());
    }

    void Add(const std::string& name, ParamType parameter)
    {
        Register(parameter->ModelNode(), name);

        parameters.emplace_back(std::move(parameter));
        paramNameMap[name] = parameters.size() - 1;
    }

protected:

    std::vector<ParamType> parameters;
    std::map<std::string, size_t> paramNameMap;
};

GdkCppFx(std::unique_ptr<ExtParam>) ConstructExtParamFromInfo(GdkParamInfo info, bool copyDefaults = true);

#endif
