#ifndef kVS_XML_UTILS_H
#define kVS_XML_UTILS_H

#include <kVision/Common/kVision.h>
#include <kFireSync/Utils/kSensorIdentity.h>
#include <kFireSync/kNodeDef.h>
#include <kApi/Io/kDirectory.h>

/** 
 * Preprocessing function which copies in, xml files that have been included using the <include href="address"/> tag.
 * This function supports both relative and absolute paths (automatically detected) and to resolve the relative paths it needs to know the path of the original xml file.
 * If an included file is not found or fails to load as a valid XML the function returns an error status.
 *
 * Returns kERROR_FORMAT if there's an issue with the xml formatting.
 * Returns kERROR_NOT_FOUND if unable to load the file pointed to be an include element.
 *
 * @param   dst              Destination xml object. Can be the same as src object (in-place operation supported)
 * @param   src              Source xml object.
 * @param   srcFilePath      Source xml file path, needed to resolve relative paths. An empty string ("") can be passed in if the client is sure there will be no relative paths.
 * @return                   Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_PreprocessIncludes(kXml dst, kXml src, const kChar* srcFilePath);

/** 
 * Prunes the branches of the XML where the attributes are defined and don't match the filter. 
 *
 * Strict filtering: if strict is set to kTRUE, any XML element that has a filter XML Attribute that is NOT contained within the filter kMap
 * will be treated as if it did not match this filter. However, elements do not need to have every attribute in the filter, only that the attributes
 * they do have are both in the filter, and match the value.
 *
 * @param   dst             Destination xml object. Can be the same as src object (in-place operation supported)
 * @param   src             Source xml object.
 * @param   filter          kMap<kString, kArrayList<kString>>. For instance {{"ModelNumber",{2340}}, {"HardwareRevision",{"A","B"}}
 * @param   strict          kTRUE to use strict filtering
 * @return                  Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_Filter(kXml dst, kXml src, kMap filter, kBool strict);

/** 
 * Goes through all XML node values and finds all substrings of the form $([field name]) and replaces them with the value defined for it in the fieldValues map.
 * if a the value for that field is not defined (not found in kMap) the function will returns a KERROR_NOT_FOUND.
 *
 * NOTE: Any macros without a matching KvP in the fieldValues map will be ignored and left in the file.
 *
 * @param   dst              Destination xml object. Can be the same as src object (in-place operation supported)
 * @param   src              Source xml object.
 * @param  fieldValues       Contains the values for macros. kMap<kString, kString>. For instance {{"ModelNumber", 2340}, {"HardwareRevision","A"}}
 * @return                   Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_ResolveMacros(kXml dst, kXml src, kMap fieldValues);

/** 
 * For all nodes with the same path, it chooses one with the most subsuming set of attributes and removes the rest, 
   leaving behind an XML object comprised of all unique paths, matching the most specific filters.
 * @param   dst              Destination xml object. Can be the same as src object (in-place operation supported)
 * @param   src              Source xml object.
 * @return                   Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_Merge(kXml dst, kXml src);

/** 
 * Adds the Key-Value pairs from the filter to the xml root element as attributes. Useful for storing the filter used for filtering
   in the xml.
 * @param  dst               Destination xml object.
 * @param  filter            kMap<kString, kString>.
 * @return                   Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_AddFilterToRootAttrs(kXml dst, kMap filter);

/** 
 * Removes all attributes that are 'filter tags', ie. start with a _
 * @param  dst               Destination xml object. Can be the same as src object (in-place operation supported)
 * @param  src               Source xml object.
 * @return                   Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_CleanFilterTags(kXml dst, kXml src);

/** 
 * Returns the full Xml path for a node.
 *
 * @param  xml              Xml object.
 * @param  node             Xml node to get the full path for.
 * @param  path             kString object (empty string) used to build the full path.
 * @return                  Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_FullPath(kXml xml, kXmlItem node, kString path);

/** 
 * Returns the path of an item relative to a given parent node. If parent node is kNull, it returns the full path.
 *
 * @param  xml              Xml object.
 * @param  node             Xml node to get the full path for.
 * @param  path             kString object (empty string) used to build the full path.
 * @return                  Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_PathFromNode(kXml xml, kXmlItem parent, kXmlItem node, kString path);

/** 
 * Returns kTrue if xml contains no unresolved macros, else kFalse
 *
 * @param  xml              Xml object.
 * @return                  Operation status. 
 */
kVsFx(kBool) kVsXmlUtils_AllMacrosResolved(kXml src);



kVsFx(kStatus) kVsXmlUtils_PreprocessFilterTags(kXml xml, kXmlItem node);

/** 
 * Returns a Map version of the XML item, with keys being paths of xml items, formatted as ROOTNAME/BRANCHNAME/.../LEAFNAME
  and values of kStrings with contents matching the ctext value of the XML leaf.

  IMPORTANT NOTE: The converter only supports XMLs with unique paths for all items, otherwise returns kERROR_ALREADY_EXISTS
  In future support for non-unique paths can be added by indexing such as :
  
  ROOTNAME/BRANCHNAME/.../LEAFNAME
  ROOTNAME/BRANCHNAME/.../LEAFNAME[0]
  ROOTNAME/BRANCHNAME/.../LEAFNAME[1]
  ROOTNAME/BRANCHNAME/.../LEAFNAME[2] ...

  With the value of ROOTNAME/BRANCHNAME/.../LEAFNAME being equal to ROOTNAME/BRANCHNAME/.../LEAFNAME[0] for backwards compatibility. 

  Other options include having the value of ROOTNAME/BRANCHNAME/.../LEAFNAME be a kArrayList of values of the leaves with the same path. Might be more elegant. 

 *
 * @param  xml              Xml object.
 * @param  xml              Xml item to be converted
 * @param  map              Output kMap object.
 * @return                  Operation status. 
 */
kVsFx(kStatus) kVsXmlUtils_XmlToMap(kXml xml, kXmlItem item, kMap* map);


kVsFx(kBool) kVsXmlUtils_MapEquals(kMap a, kMap b);

/* Extracts the first macro $(X)->X that is found in the text
 *
 * @param  text              text that has possible multiple macros
 * @param  startindex        index from which start search
 * @param  macro             output macro
 * @param  capacity          capacity of output
 * @return                   Operation status. 
 */
kVsFx(kStatus)  kVsXmlUtils_FindMacroAt(const kChar* text, kSize startIndex, kChar* macro, kSize capacity);

/* replaces the first macro, X->V $(X)->V, that is found in the text
 *
 * @param  text              text that has possible multiple macros
 * @param  startindex        index from which start search
 * @param  macroValue        value which will replace the first macro
 * @param  capacity          capacity of output
 * @return                   Operation status. 
 */
kVsFx(kStatus)  kVsXmlUtils_ReplaceMacroAt(const kChar* macroValue, kChar* text, kSize startIndex, kSize capacity);

#include <kVision/Vs/kVsXmlUtils.x.h>

#endif  /* kVS_XML_UTILS_H */
