/**
 * @file    kVsSpotExperimental.h
 * @brief   Declares experimental spot detection structures and functions.
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_VISION_SPOT_EXPERIMENTAL_H
#define K_VISION_SPOT_EXPERIMENTAL_H

#include <kFireSync/Data/kSpot.h>

#include <kVision/Common/kVision.h>

#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kImage.h>

/**
 * Savitzky-Golay filtered Zero crossing spot detector
 *
 * @param  image       Input image. Must be of type k8u.
 * @param  edgeWindow  Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  fitOrder    Savitzky-Golay polynomial order
 * @param  threshold   Rising and falling edge threshold
 * @param  spots       kArrayList of kSpot. Will be populated with detected spots
 * @param  edgeImage   kImage of type k8s. Will be populated with edge image if edgeImage != kNULL
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SavitzkyGolayZC(kImage image, k32u edgeWindow, k32u fitOrder, k32u threshold,
    kArrayList spots, kImage edgeImage);

/**
 * Savitzky-Golay filtered center of gravity spot detector.
 * Savitzky-Golay filtering is used for spot entrance and exit detection
 *
 * @param  image           Input image. Must be of type k8u.
 * @param  edgeWindow      Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  fitOrder        Savitzky-Golay polynomial order
 * @param  threshold       Rising and falling edge threshold
 * @param  widthThreshold  Used for width calculation, only pixels with an intensity greater than baseIntensity + widthThreshold are accepted.
 * @param  minWidth        minimum spot width
 * @param  maxWidth        maximum spot width
 * @param  minSum          minimum spot sum
 * @param  spots           kArrayList of kSpot. Will be populated with detected spots
 * @param  bounds          kArrayList of kPoint32s. If not null, will be populated with column boundaries
 * @param  edgeImage       kImage of type k8s. Will be populated with edge image if edgeImage != kNULL
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SavitzkyGolayAccg(kImage image, k32u edgeWindow, k32u fitOrder, k32u threshold,
    k32u widthThreshold, k32u minWidth, k32u maxWidth, k32u minSum, kArrayList spots, kArrayList bounds, kImage edgeImage);

/**
 * Sobel filtered Zero-Crossing edge detection
 *
 * @param  image          Input image. Can be of type k8u or k64f
 * @param  averageWindow  Number of columns to average over for Sobel filter
 * @param  edgeWindow     Window size to use for (Savitzky-Golay) best fit derivative filter
 * @param  threshold      Rising and falling edge threshold
 * @param  spots          kArrayList of kSpot. Will be populated with detected spots
 * @param  edgeImage      kImage of type k8s. Will be populated with edge image if edgeImage != kNULL
 * @param  balance        Whether gradients should be balanced.
 * @returns kStatus
 */
kVsFx(kStatus) kVsSpotExp_SobelZC(kImage image, k32u averageWindow, k32u edgeWindow, k32u threshold,
    kArrayList spots, kImage edgeImage, kBool balance);

#include <kVision/Vs/kVsSpotExperimental.x.h>

#endif
