/**
* @file kVsSpline.h
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
*/

#ifndef K_VISION_SPLINE_H
#define K_VISION_SPLINE_H

#include <kApi/kApiDef.h>
#include <kVision/Common/kVision.h>

/**
* @class       kVsSpline
* @extends     kObject
* @ingroup     kVision-Spline
* @brief       Used to fit and apply splines.
*/
typedef kObject kVsSpline;

typedef enum kVsSplineType
{
    kVS_SPLINE_LINEAR = 1,
    kVS_SPLINE_CATMULL_ROM,
} kVsSplineType;

/**
 * Constructs a kVsSpline object.
 *
 * @public              @memberof kVsSpline
 * @param   spline      Destination for the constructed object handle.
 * @param   knotCount   Optionally specifies the number of spline points for which to allocate space (can be zero). 
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status.
*/
kVsFx(kStatus) kVsSpline_Construct(kVsSpline* spline, kSize knotCount, kAlloc allocator);

/**
 * Calculates the output of the spline function for a single input value. 
 *
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @param   x           Input value.
 * @return              Spline output value.
*/
kVsFx(k64f) kVsSpline_Apply(kVsSpline spline, k64f x); 

/**
 * Calculates outputs of the spline function for an array of input values.
 *
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @param   x           Input values.
 * @param   f           Spline output values.
 * @param   count       Count of input/output values.
 * @return              Operation status.
*/
kVsFx(kStatus) kVsSpline_ApplyN(kVsSpline spline, const k64f* x, k64f* f, kSize count);

/**
 * Sets the valid domain for the spline.
 *
 * If domain limits are set and an input value provided to the Apply (or ApplyN) function 
 * is outside of the domain, the output will be k64F_NULL.
 * 
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @param   xMin        Minimum domain value.
 * @param   xMax        Maxium domain value.
 * @return              Operation status.
*/
kVsFx(kStatus) kVsSpline_SetDomain(kVsSpline spline, k64f xMin, k64f xMax); 

/**
 * Gets the polynomial coefficients at the specified index. 
 *
 * During data fitting, one polynomial is generated for each adjacent pair of unique x-values. 
 * Accordingly, the number of generated polynomials will be one less than the knot count.  
 * 
 * The order of the polynomials is determined by the data-fitting function. N+1 coefficients 
 * will be generated per polynomial for an order N data fit. 
 *
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @param   index       Polynomial index. 
 * @return              Pointer to the requested polynomial coefficients (lowest-order coefficient first). 
*/
kVsFx(k64f*) kVsSpline_Coefficients(kVsSpline spline, kSize index); 

/**
 * Reports the count of spline knots.
 *
 * The number of spline knots is determined by the number of unique x-values provided 
 * during data fitting.
 *
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @return              Knot count.
*/
kVsFx(kSize) kVsSpline_KnotCount(kVsSpline spline); 

/**
* Gets the x-values corresponding to the spline knots.
*
* @public              @memberof kVsSpline
* @param   spline      Spline object.
* @return              Pointer to spline knots. 
*/
kVsFx(k64f*) kVsSpline_Knots(kVsSpline spline); 

/**
 * Gets the user-defined minimum x-value for the spline.
 *
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @return              Minimum x value.
*/
kVsFx(k64f) kVsSpline_DomainMin(kVsSpline spline); 

/**
* Gets the user-defined maximum x-value for the spline.
*
* @public              @memberof kVsSpline
* @param   spline      Spline object.
* @return              Maximum x value.
*/
kVsFx(k64f) kVsSpline_DomainMax(kVsSpline spline); 

/**
 * Generates spline coefficients for the given input data. 
 * 
 * The input data will be used to generate spline coefficients. The input data points 
 * will first be sorted by x-value. One spline knot will then be created for each 
 * unique x-value.  At least two unique values are required.  
 * 
 * @public              @memberof kVsSpline
 * @param   spline      Spline object.
 * @param   type        kVsSplineType of fitting polynomial segments.
 * @param   x           x (input) values.
 * @param   f           f (output) values.
 * @param   count       Count of fit points. 
 * @return              Operation status.
*/
kVsFx(kStatus) kVsSpline_Fit(kVsSpline spline, kVsSplineType type, const k64f* x, const k64f* f, kSize count);

#include <kVision/Vs/kVsSpline.x.h>

#endif /* #ifndef K_VISION_SPLINE_H */

