/**
* @file     kVsBlackLevelAlg.h
* @brief    Declares the BlackLevelAlg class
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kVS_BLACK_LEVEL_ALG_H
#define kVS_BLACK_LEVEL_ALG_H

#include <kVision/Common/kVision.h>

;

/**
* @class       kVsBlackLevelAlg
* @extends     kObject
* @ingroup     kVision-Vs
* @brief       Implementation of the camera Black Level calibration algorithm.
*/
typedef kObject kVsBlackLevelAlg;

/**
* Constructs a kVsBlackLevelAlg object.
*
* @public           @memberof kVsBlackLevelAlg
* @param    alg     Destination for the constructed object handle
* @return           Operation Status
*/
kVsFx(kStatus) kVsBlackLevelAlg_Construct(kVsBlackLevelAlg* alg, kAlloc allocator);

//-- accessors ---------

/**
* @brief Sets the target ground intensity for black level calibration.
* 
* The target intensity is the (theoretical) intensity value that corresponds to the
* optimal black level in the linear relationship: blackLevel = A * intensity + B. 
* During calibration A and B are determined empirically, and the optimal black level
* is determined as follows: optimalBlackLevel = A * targetGround + B.
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetTargetGround(kVsBlackLevelAlg alg, k64f ground);

/**
* @brief Gets the target ground intensity.
* @see kVsBlackLevelAlg_SetTargetGround
*/
kVsFx(k64f) kVsBlackLevelAlg_TargetGround(kVsBlackLevelAlg alg);

/**
* @brief Sets the intensity ground threshold for ground fault pixel verification.
*
* Ground fault pixels are defined as pixels with an intensity out of range of the
* target ground value by more than the ground fault threshold when the black level is
* set to its optimal value. Many imagers possess many pixels whose intensity does not
* reach the target ground value under optimal black level settings. This threshold
* defines how far from the target value those pixels can be before they are considered
* "ground fault" (defective).
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetGroundThreshold(kVsBlackLevelAlg alg, k64f threshold);

/**
* @brief Gets the ground intensity threshold.
* @see kVsBlackLevelAlg_SetGroundThreshold
*/
kVsFx(k64f) kVsBlackLevelAlg_GroundThreshold(kVsBlackLevelAlg alg);

/**
* @brief Sets the minimum permitted black level [0,255].
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetBlackLevelMin(kVsBlackLevelAlg alg, k32s blackLevelMin);

/**
* @brief Gets the minimum permitted black level.
*/
kVsFx(k32u) kVsBlackLevelAlg_BlackLevelMin(kVsBlackLevelAlg alg);

/**
* @brief Sets the maximum permitted black level [0,255].
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetBlackLevelMax(kVsBlackLevelAlg alg, k32s blackLevelMax);

/**
* @brief Gets the maximum permitted black level [0,255].
*/
kVsFx(k32u) kVsBlackLevelAlg_BlackLevelMax(kVsBlackLevelAlg alg);

/**
* @brief Sets the slope boundary percentage for stuck pixel detection [0,1].
*
* During stuck pixel verification, the slope term A of the relationship: blackLevel = A * intensity + B
* is determined for each pixel. Pixels are considered "slope outliers" if their slope A differs from the
* mean A' of all pixels by more than the specified percentage of the mean. Percentages are expressed
* as values in teh range [0,1]. For example, if the mean slope is 4.0 and the slopeBoundaryPercent 
* is 0.1 (10%) then the acceptable slope range is [3.6, 4.4].
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetSlopeBoundPercent(kVsBlackLevelAlg alg, k64f slopeBoundPercent);

/**
* @brief Get the slope boundary percentage.
* @see kVsBlackLevelAlg_SetSlopeBoundPercent
*/
kVsFx(k64f) kVsBlackLevelAlg_SlopeBoundPercent(kVsBlackLevelAlg alg);

/**
* @brief Set adjacency threshold(s) for x and y axes.
*
* Two pixels are considered adjacent if the x and y components of the vector between them
* are both within the adjacency thresholds. Effectively, this defines a bounding box centered
* on one pixel with size 2x+1,2y+1. Another pixel is considered adjacent if it falls into this
* bounding box (inclusive).
*/
kVsFx(kStatus) kVsBlackLevelAlg_SetAdjacencyThreshold(kVsBlackLevelAlg alg, kPoint32s adjacencyThreshold);

/**
* @brief Get adjacency thresholds.
* @see kVsBlackLevelAlg_SetAdjacencyThreshold
*/
kVsFx(kPoint32s) kVsBlackLevelAlg_AdjacencyThreshold(kVsBlackLevelAlg alg);

/**
* @brief Computes the optimal black level from a series of images matched with the corresponding
* black level.
*
* The image ground intensity (when exposure is at minimum) varies (almost perfectly) linearly by
* the black level setting. The optimal black level is defined based on that linear relationship
* blackLevel = A * intensity  +B. ComputeOptimalBlack level takes a set of N images with corresponding
* black level values, and computes the mean intensity of each image. It then performs a linear
* regression to determine A and B. Given a target ground intensity (targetGround) the optimal
* black level is determined by: optimalBlackLevel = A * targetGround + B. The optimal black level is
* then clamped to the range [blackLevelMin, blackLevelMax].
*
* @param    alg                     The algorithm object
* @param    images                  An array of kImage objects
* @param    blackValues             An array of k32u blackLevel values corresponding to the images
* @param    count                   The length of both the "images" and "blackValues" arrays. Must be >= 2
* @param    fitSlope                Output parameter: will be set to the slope of the linear fit
* @param    fitIntercept            Output parameter: will be set to the intercept of the linear fit
* @param    result                  Output parameter: will be set to the optimal black level
* @param    intensityDataFiltered   Output parameter: will be populated with the x,y pairs corresponding to the intensity vs black level.
* 
* @see kVsBlackLevelAlg_SetTargetGround
* @see kVsBlackLevelAlg_SetBlackLevelMin
* @see kVsBlackLevelAlg_SetBlackLevelMax
*/
kVsFx(kStatus) kVsBlackLevelAlg_ComputeOptimalBlackLevel(kVsBlackLevelAlg alg, const kImage* images, const k32s* blackValues, k32u count, k64f* fitSlope, k64f* fitIntercept, k32s* result, kArrayList intensityDataFiltered);

/**
* @brief Computes an intensity histogram of a kImage<k8u>.
*
* Counts the number of pixels in a kImage<k8u> at each of the 256 possible intensity values. This is
* used to characterize the result of black level calibration by showing the distribution of pixel
* intensities.
*/
kVsFx(kStatus) kVsBlackLevelAlg_ComputeIntensityHistogram(kVsBlackLevelAlg alg, kImage image, k64u* histogram, kSize histogramCount);

/**
* @brief Computes the number of pixels out of range of the target ground by more than the ground threshold based on an intensity histogram.
*
* Sums the pixel counts in all binds which are out of range of the target ground by more than the groundThreshold field of the 
* kVsBlackLevelAlg alg.
*
* @param    alg             The black level algorithm (containing the groundThreshold)
* @param    histogram       The histogram of pixel counts, where the index corresponds to the pixel
*                           intensity.
* @param    histogramCount  The number of bins in the histogram.
* @param    count           Output parameter: will be filled with the final pixel count.
* @return   Returns the operation status.
*/
kVsFx(kStatus) kVsBlackLevelAlg_GroundFaultPixelCount(kVsBlackLevelAlg alg, const k64u* histogram, kSize histogramCount, k64u* count);

/**
* @brief Compute the linear relationship between the ground intensity and the black level on a 
* per-pixel basis.
*
* Based on a set of N images (N > 1) and their corresponding black level values, this method 
* determines the slope and intercept (A,B) parameters of the linear relationship 
* blackLevel = A * intensity + B for each pixel.
*
* @param    alg             The algorithm object
* @param    images          An array of kImage objects, expected to contain ground images (minimum exposure)
*                           of equal size, pixel type k8u.
* @param    blackValues     An array of black level values in range [0,255] corresponding to the ground images.
* @param    count           The number of black level values and ground images.
* @param    slopeMap        A kArray2<k64f> of size matching the images. Note that image sizes are expressed
*                           as width x height but array sizes are expressed rows x columns (transposed). This
*                           array will be filled with the slope (A) term for the linear fit for each pixel.
* @param    interceptMap    A kArray2<k64f> of size matching the images. This array will be filled with the 
*                           intercept (B) term of the linear fit for each pixel.
* @return   Returns the operation status. kERROR_PARAMETER indicates a mismatch in the image or array dimensions.
*/
kVsFx(kStatus) kVsBlackLevelAlg_ComputePixelResponse(kVsBlackLevelAlg alg, const kImage* images, const k32u* blackValues, k32u count, kArray2 slopeMap, kArray2 interceptMap);

/**
* @brief Determines which pixels display an abnormal linear relationship between ground intensity and black level
* based on the slope,intercept values per pixel generated by kVsBlackLevelAlg_ComputePixelResponse.
*
* The slope,intercept values for each pixel permit the computation of a centroid value. Pixels are considered
* "outliers" *abnormal" if their slope differs from the mean by "slopeBoundPercent", OR if the linear relationship
* indicates that no valid [0,255] black level exists that would permit a ground intensity below the ground threshold.
*
* @see  kVsBlackLevelAlg_ComputePixelResponse
*
* @param  alg               The algorithm object, from which the groundThreshold, blackLevelMin, and slopeBoundPercent
*                           are used in this operation.
* @param  slopeMap          A kArray2<k64f> representing the slope of the linear relationship for each pixel.
* @param  interceptMap      A kArray2<k64f> representing the intercept of the linear relationship for each pixel. Its 
*                           dimensions must match those of slopeMap.
* @param  blackLevel        The current black level
* @param  centroid          Output parameter: will be filled with the [x,y] = [slope,intercept] centroid of the data
*                           in slopeMap, interceptMap.
* @param  basisA            Output parameter: will be filled with the basis vector of the major axis of the covariance
*                           ellipse of the slope,intercept data.
* @param  basisB            Output parameter: will be filled with the basis vector of the minor axis of the covariance
*                           ellipse of the slope,intercept data.
* @param  covarianceMatrix  Output parameter: must be a 2x2 kArray2<k64f>, whose values will be filled with the covariance
*                           matrix of the slope,intercept data.
* @param  stuckPixelMap     Output parameter: must be a kArray2<k8u> of dimensions matching slopeMap and interceptMap. Will
*                           be filled with zeroes, except for outlier pixels, which will be set to value 255.
* @param  stuckPixelCount   Output parameter: will be filled with the number of outlier pixels.
* @return Returns the operation status. kERROR_PARAMETER indicates a mismatch in the image or array dimensions.
*/
kVsFx(kStatus) kVsBlackLevelAlg_ComputeStuckPixelMap(kVsBlackLevelAlg alg, kArray2 slopeMap, kArray2 interceptMap, kPoint64f* centroid, kPoint64f* basisA, kPoint64f* basisB, kArray2 covarianceMatrix, kArray2 stuckPixelMap, k32u* stuckPixelCount);

/**
* @brief Counts the number of stuck pixels within a fixed-size bounding box of another stuck pixel based on the stuck
* pixel map created by kVsBlackLevelAlg_ComputeStuckPixelMap.
*
* Stuck pixels in close proximity to each other have a more pronounced effect on image quality. This method determines
* how many stuck pixels in the stuckPixelMap are within a pair of threshold on their x and y displacement. This effectively
* draws a fixed-size bounding box centered on each stuck pixel, and marks that pixel as "adjacent" if another pixel fits in 
* its bounding box. Each pixel is counted exactly once, regardless of the number of adjacent neighbours it has. The method
* also draws a colour (kRgb) visualization where stuck pixels are white and adjacent pixels are red. The thresholds in both 
* x and y for adjacency are determined by the algorithm field adjacencyThreshold (type kPoint32s).
*
* @see kVsBlackLevelAlg_ComputeStuckPixelMap
*
* @param    alg                 The algorithm object, from which the adjacencyThreshold field is used.
* @param    stuckPixelMap       The stuckPixelMap (kArray2<k8u>) object produced by kVsBlackLevelAlg_ComputeStuckPixelMap.
* @param    adjacentPixelCount  Output parameter: filled with the number of pixels that meet the adjacency criteria.
* @param    adjacentPixelMap    Output parameter: must be a kImage<kRgb> of dimensions match the stuckPixelMap. Will be
*                               filled with zeroes, except for stuck pixels which will be white and adjacent pixels which
*                               will be red.
* @return Returns the operation status.
*/
kVsFx(kStatus) kVsBlackLevelAlg_CountAdjacentStuckPixels(kVsBlackLevelAlg alg, kArray2 stuckPixelMap, k32u* adjacentPixelCount, kImage adjacentPixelMap);

/**
* @brief Performs the same visualization conversion from k8u -> kRgb as performed by kVsBlackLevelAlg_CountAdjacentStuckPixels,
* but without checking for pixel adjacency.
*
* In the event that there are a large number of stuck pixels, it is both unnecessary and computationally expensive to 
* perform the adjacency detection. Given that that verification will have already failed, this method is used to generate
* the visualization without the additional cost in that case.
*
* @param    alg                 The algorithm object, from which the adjacencyThreshold field is used.
* @param    stuckPixelMap       The stuckPixelMap (kArray2<k8u>) object produced by kVsBlackLevelAlg_ComputeStuckPixelMap.
* @param    stuckPixelImage     Output parameter: must be a kImage<kRgb> of dimensions match the stuckPixelMap. Will be
*                               filled with zeroes, except for stuck pixels which will be white.
* @return Returns the operation status.
*/
kVsFx(kStatus) kVsBlackLevelAlg_VisualizeStuckPixels(kVsBlackLevelAlg alg, kArray2 stuckPixelMap, kImage stuckPixelImage);

/**
* @brief Checks if a particular slope,intercept pair qualifies as an outlier.
*
* In this case, an outlier is defined as any slope,intercept pair where EITHER the slope differs from the slopeMean
* by an (absolute) value greater that slopeBoundPercent*slopeMean OR the value (blackLevelMin - intercept) / slope
* is greater than groundThreshold.
*
* @see kVsBlackLevelAlg_ComputeStuckPixelMap
*
* @param    alg         The algorithm object, from which the blackLevelMin, groundThreshold, and slopeBoundPercent
*                       fields are used.
* @param    slopeMean   The mean slope of all pixels (from the centroid).
* @param    interceptMean   The mean intercept of all pixels (from the centroid).
* @param    slope       The slope of a particular pixel's linear relationship.
* @param    intercept   The intercept of a particular pixel's linear relationship.
* @return Returns kTRUE if the slope,intercept pair is an outlier.
*/
kVsFx(kBool) kVsBlackLevelAlg_IsOutlier(kVsBlackLevelAlg alg, k64f slopeMean, k64f interceptMean, k64f slope, k64f intercept);

;

#include <kVision/Vs/kVsBlackLevelAlg.x.h>

#endif /* #ifndef kVS_BLACK_LEVEL_ALG_H */