/**
* @file    kS3dVolumeCheckInputAlg.h
* @brief   Declares the kS3dVolumeCheckInputAlg class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_VOLUMECHECKINPUTALG_H
#define KVISION_S3D_VOLUMECHECKINPUTALG_H

#include <kVision/S3d/kS3dVolumeCheckDef.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dVolumeCheckPlotter.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dOmcFile.h>
#include <kApi/Io/kDirectory.h>
#include <kApi/Threads/kTimer.h>

#define kS3D_VOLUMECHECKTEST_INPUT_FILENAME          "%s%d.kdat"
#define kS3D_VOLUMECHECKTEST_INPUT_INFOFILENAME      "%sInfo%d.kdat"
#define kS3D_VOLUMECHECKTEST_INPUT_OMCFILENAME       "%s%d.omc"

#define kS3D_VOLUMECHECKTEST_INPUT_FILE_MAXCOUNT     (10)

/**
* @class       kS3dVolumeCheckInputAlg
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides input data for inputAlging algorithms of kS3dVolumeCheckAlg.
*/
typedef kObject kS3dVolumeCheckInputAlg;

/**
* Constructs a kS3dVolumeCheckInputAlg object
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_Construct(kS3dVolumeCheckInputAlg* inputAlg, kAlloc allocator);

/**
* Load input height map or point clouds.
*
* The accepted simulated input kArray2 type are:
*
*     kArray2 < kPoint3d16s >   "*.kdat"     Manipulation will be performed: transformation of data with kS3dTransform6fd pos.
*     kArray2 < kPoint3d32s >   "*.kdat"     Manipulation will be performed: transformation of data with kS3dTransform6fd pos.
*     kArray2 < kPoint3d32f >   "*.kdat"     Manipulation will be performed: transformation of data with kS3dTransform6fd pos.
*     kArray2 < kPoint3d64f >   "*.kdat"     Manipulation will be performed: transformation of data with kS3dTransform6fd pos.

*     kArray2 < k16s >          "*.kdat"     Manipulation will not be performed.
*     kArray2 < k32s >          "*.kdat"     Manipulation will not be performed.
*     kArray2 < k32f >          "*.kdat"     Manipulation will not be performed.
*     kArray2 < k64f >          "*.kdat"     Manipulation will not be performed.
*  
* For omc input file, the data will be converted to the specified format.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg
* @param   data        Output height map or point cloud array.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_LoadInputFile(kS3dVolumeCheckInputAlg inputAlg, kArray2* data);

/**
* Get the number of recorded measurements.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              number of recorded measurements.
*/
kVsFx(k32u) kS3dVolumeCheckInputAlg_MeasurementIndex(kS3dVolumeCheckInputAlg inputAlg);

/**
* Get scale coefficients for input point clouds or height map.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Scale coefficients.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputAlg_ActualScale(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set scale coefficients for input point cloud or height map.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   scale       Scale coefficients.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetInputScale(kS3dVolumeCheckInputAlg inputAlg, kPoint3d64f scale);

/**
* Save chamber type and scale coefficients of the input point cloud or height map.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   fileName    file name ".kdat".
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SaveSimulationInputInfo(kS3dVolumeCheckInputAlg inputAlg, const kChar* fileName);

/**
* Get type of combustion chamber, defined in kS3dVolumeCheckDefaultChamberType.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Type of the combustion chamber.
*/

kVsFx(k32u) kS3dVolumeCheckInputAlg_ChamberType(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set type of combustion chamber, defined in kS3dVolumeCheckDefaultChamberType.
*
* @public               @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg     kS3dVolumeCheckInputAlg object.
* @param   chamberType  Type of the combustion chamber.
* @return               Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetChamberType(kS3dVolumeCheckInputAlg inputAlg, k32u chamberType);

/**
* Get basic file name of the simulated input file.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Basic file name.
*/
kVsFx(kChar*) kS3dVolumeCheckInputAlg_FileName(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set basic file name of the simulated input file.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   fileName    Basic input file name.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetFileName(kS3dVolumeCheckInputAlg inputAlg, kChar* fileName);

/**
* Get format of the simulated input data defined in kS3dVolumeCheckInputFormatType.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Basic input file name.
*/
kVsFx(k32s) kS3dVolumeCheckInputAlg_DataFormat(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set format of the simulated input data defined in kS3dVolumeCheckInputFormatType.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   format      Format of the simulated input data
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetDataFormat(kS3dVolumeCheckInputAlg inputAlg, k32s format);

/**
* Get available file count of the simulattion for a given data format.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Available file count.
*/
kVsFx(k32s) kS3dVolumeCheckInputAlg_FileCount(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set available file count of the simulattion for a given data format.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   count       Available file count.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetFileCount(kS3dVolumeCheckInputAlg inputAlg, k32s count);

/**
* Get start index for several measurements. e.g. index=0: corresponds to input0.omc.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Start index for several measurements.
*/
kVsFx(k32s) kS3dVolumeCheckInputAlg_StartIndex(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set start index for several measurements. e.g. index=0: corresponds to input0.omc.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   startIndex  Start index for several measurements
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetStartIndex(kS3dVolumeCheckInputAlg inputAlg, k32s startIndex);

/**
* Get measurement count for several measurements. If the count is set more than available file count, the file is loaded into the loop.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Measurement count.
*/
kVsFx(k32s) kS3dVolumeCheckInputAlg_MeasurementCount(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set measurement count for several measurements. If the count is set more than available file count, the file is loaded into the loop.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   count       Measurement count.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetMeasurementCount(kS3dVolumeCheckInputAlg inputAlg, k32s count);

/**
* Get actual position of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Start position.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputAlg_ActualPosition(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set start position of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   position    Start position.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetStartPosition(kS3dVolumeCheckInputAlg inputAlg, kPoint3d64f position);

/**
* Get actual orientation of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Start orientation.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputAlg_ActualOrientation(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set start orientation of the simulated transformation of the surface.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   orientation Start orientation.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetStartOrientation(kS3dVolumeCheckInputAlg inputAlg, kPoint3d64f orientation);

/**
* Queries whether the transformation of the surface with the random generator is performed.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Random transformation will be performed?
*/
kVsFx(kBool) kS3dVolumeCheckInputAlg_RandomPositioning(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set a boolean value to determine whether the transformation of the surface with the random generator should be performed.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @param   randomUsed  kTRUE = will be performed
* @return              Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetRandomPositioning(kS3dVolumeCheckInputAlg inputAlg, kBool randomUsed);

/**
* Get displacement amplitudes of the random generator.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Displacement amplitudes in x, y and directions.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputAlg_RandomPosition(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set displacement amplitudes of the random generator.
*
* @public                  @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg        kS3dVolumeCheckInputAlg object.
* @param   randomPosition  Displacement amplitudes in x, y and directions.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetRandomPosition(kS3dVolumeCheckInputAlg inputAlg, kPoint3d64f randomPosition);

/**
* Get Rotation amplitudes of the random generator.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              Rotation amplitudes of the 3 angles.
*/
kVsFx(kPoint3d64f) kS3dVolumeCheckInputAlg_RandomOrientation(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set Rotation amplitudes of the random generator.
*
* @public                     @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg           kS3dVolumeCheckInputAlg object.
* @param   randomOrientation  Rotation amplitudes of the 3 angles.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetRandomOrientation(kS3dVolumeCheckInputAlg inputAlg, kPoint3d64f randomOrientation);

/**
* Queries whether the loaded omc file should be stored as kdat file used for further simulation.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              kTrue = kdat file will be stored.
*/
kVsFx(kBool) kS3dVolumeCheckInputAlg_StoreInputData(kS3dVolumeCheckInputAlg inputAlg);

/**
* Set a boolean value to determine whether loaded omc file should be stored as kdat file used for further simulation.
*
* @public                  @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg        kS3dVolumeCheckInputAlg object.
* @param   saveSimulation  kTrue = kdat file will be stored.
* @return                  Operation status.
*/
kVsFx(kStatus) kS3dVolumeCheckInputAlg_SetStoreInputData(kS3dVolumeCheckInputAlg inputAlg, kBool saveSimulation);

/**
* Queries whether the additional infoation exists.
*
* @public              @memberof kS3dVolumeCheckInputAlg
* @param   inputAlg    kS3dVolumeCheckInputAlg object.
* @return              kTrue = the additional infoation exists.
*/
kVsFx(kBool) kS3dVolumeCheckInputAlg_InfoExists(kS3dVolumeCheckInputAlg inputAlg);

#include <kVision/S3d/kS3dVolumeCheckInputAlg.x.h>

#endif /* #ifndef KVISION_S3D_VOLUMECHECKINPUTALG_H */  
