/**
* @file    kS3dPolynomialMap.h
* @brief   Declares the kS3dPolynomialMap class. 
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_POLYNMAP_H
#define KVISION_S3D_POLYNMAP_H

#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/S3d/kS3dPolynomial.h>

#include <kApi/kApiDef.h>
#include <kApi/kAlloc.h>
#include <kVision/Common/kVision.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kString.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <math.h>
#include <time.h>
#include <stdio.h>

/**
* @class       kS3dPolynomialMap
* @extends     kObject
* @ingroup     kVision-M3d
* @brief       Provides a set of utility functions for calculating polynomial function from a given image.
*/
typedef kObject kS3dPolynomialMap;

/**
* Constructs a kS3dPolynomialMap object
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_Construct(kS3dPolynomialMap* polynMap, kAlloc allocator);

/**
* Calculates polynomial function from a data set with full surfaces.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Output plynomial function. 
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynMap(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Calculates polynomial function from a data set with marked area.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.
* @param   polynomial  Output plynomial function.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_ValidPolynMap(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid, kS3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Harmonization of light-intensity variation by dividing the best-fit polynomial function. Only for grid image.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Plynomial function.
* @param   uniformity  Handle of k64f to store the Uniformity of light-intensity variation.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @param   invers      Determines whether the image should be inverted.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_NormalizeGridImage(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynomial, k64f* uniformity, kS3dArray2DataType type, kBool invers);

/**
* Iterative computing the polynomial function to extrapolate the missing data according to existing data.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Plynomial function.
* @param   order0      Start order of the plynomial.
* @param   order       Target order of the plynomial.
* @param   repeat      repeat count.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_RobustValidPolynMap(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid,
    kS3dPolynomial polynomial, k32s order0, k32s order, k32s repeat,  kS3dArray2DataType type);


/**
* Calculation of the polynomial function from the derivative image.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.
* @param   polynomial  Plynomial function.
* @param   wavelength  Equivalent wavelength of pattern.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_Derivative3orderMap(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid, kS3dPolynomial polynomial, k32f   wavelength, kS3dArray2DataType type);

/**
* Set order of the polynomial.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   order       order of the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_SetOrder(kS3dPolynomialMap polynMap, k32s order);

/**
* Set the second order of the polynomial function. It is very rarely used. The extrapolation of invalid region is executed with lowered polynomial
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   order       second order of the polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_SetOrder2(kS3dPolynomialMap polynMap, k32s order);

/**
* Set step for fitting polynomial.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   step        step for fitting polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_SetStep(kS3dPolynomialMap polynMap, k32s step);

/**
* Calculates the radial symmetry of the distortion distribution
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Plynomial function.
* @param   Symmetry    Handle of k64f to store the radial symmetry of the distortion distribution.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_DistortionSymmetry(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynomial, k64f* Symmetry, kS3dArray2DataType type);

/**
* Draws the polynomial function in a map.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   polynomial  Plynomial function.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_DrawPolynomial(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynomial, kS3dArray2DataType type);

/**
* Calculates polynomial function from an array 
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data. kArray1 of kPoint3d32f.
* @param   polynomial  Plynomial function.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynArray1Point3D(kS3dPolynomialMap polynMap, kArray1 data, kS3dPolynomial polynomial, kBool balance);

/**
* Calculates polynomial function from an data map with appropriate weighting for each point.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data. kArray2 of kPoint3d32f. 
* @param   polynomial  Plynomial function.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynWeightedPoint3D(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynomial);


/**
* Converts the polynomial function with new polynomial coordination.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Data buffer.  kArray2 < kPoint3d32f >.
* @param   polynOutX   Output polynomial function in x direction.
* @param   polynOutY   Output polynomial function in y direction.
* @param   polynX      Polynomial function of Coordinate x.
* @param   polynY      Polynomial function of Coordinate Y.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_ConvertPolynomialFunctions(kS3dPolynomialMap polynMap, kArray2 data, kS3dPolynomial polynOutX, kS3dPolynomial polynOutY, kS3dPolynomial polynX, kS3dPolynomial polynY);

/**
* Converts the polynomial function with new polynomial coordination.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   output      Output polynomial function.
* @param   polynZ      Input polynomial function. x and y are related to camera pixel.
* @param   polynX      Polynomial function of Coordinate x.
* @param   polynY      Polynomial function of Coordinate Y.
* @param   gScale      Grid spacing in mm.
* @param   imageSizeX  Data size in x direction.
* @param   imageSizeY  Data size in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynFromPolynCoord(kS3dPolynomialMap polynMap, kS3dPolynomial output, kS3dPolynomial polynZ, kS3dPolynomial polynX, kS3dPolynomial polynY, k64f gScale, k32s imageSizeX, k32s imageSizeY);

/**
*  Calculates the scaling factor in the form of a polynomial function based on the known groove depth.
*
* @public                @memberof kS3dPolynomialMap
* @param   polynMap      kS3dPolynomialMap object.
* @param   data          kArray2 of kPoint3d32f.
* @param   valid         Area-marking.
* @param   polynomial    Output plynomial function.
* @param   depth         Groove depth.
* @param   invalidRatio  Handle of k64f to store invalid ratio. 
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_GroovePhaseScalingX(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid, kS3dPolynomial polynomial, k64f depth, k64f* invalidRatio);

/**
*  Draws the deviation map between measured and best-fit surface.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        kArray2 of kPoint3d32f.
* @param   polynomial  Input plynomial function.
* @param   inType      Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @param   outType     Value of kS3dArray2DataType. Space for output.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_DrawPolynomialDifference(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid,
    kS3dPolynomial polynomial, kS3dArray2DataType inType, kS3dArray2DataType outType);

/**
*  The Coordinate in x direction will be changed.
*  xout(xin, y) = polynX(xin, y);
*  xin(xout, y) = output(xout,y);
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   polynX      Input plynomial function.
* @param   output      Output plynomial function.
* @param   imageSizeX  Data size in x direction.
* @param   imageSizeY  Data size in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_ChangePolynCoordX(kS3dPolynomialMap polynMap, kS3dPolynomial output, kS3dPolynomial polynX, k32s imageSizeX, k32s imageSizeY);

/**
*  Calculating the difference value of the specified component of input data.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2.
* @param   valid       Area-marking.     
* @param   polynomial  Input plynomial function.
* @param   output      Output plynomial function.
* @param   imageSizeX  Data size in x direction.
* @param   type        Value of kS3dArray2DataType. Determins the type and involved component of input data.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_MapDifference(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid, kS3dPolynomial polynomial, kS3dArray2DataType type);

/**
*  Polynomial Best-Fit with weighting.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input data of kArray2 <kPoint3d32f>.  The weight factor for each point is stored in data[].z. 
* @param   valid       Area-marking. of kArray2 <k8u>.
* @param   polynomial  Output plynomial function.
* @param   bit         Only points are included in the calculation corresponding to the marking map valid.
* @param   bX          Should the data be calculated in x direction?
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynWeightedValidPoint3D(kS3dPolynomialMap polynMap, kArray2 data, kArray2 valid, kS3dPolynomial polynomial, k8u bit, kBool bX);

/**
*  Polynomial Best-Fit in x and y directions in same time from a grid input data. 
*  The Output is corresponding to the order of grid with a shift of center.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   grid        Input data of kArray2 <kPoint64f>.  Invalid points: k64F_NULL
* @param   valid       Area-marking. of kArray2 <k8u>.
* @param   polynX      Output plynomial function in x direction.
* @param   polynY      Output plynomial function in y direction.
* @param   centerX     Grid center in x direction.
* @param   centerY     Grid center in y direction.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_PolynFromGridPoints(kS3dPolynomialMap polynMap, kArray2 grid, kS3dPolynomial polynX, kS3dPolynomial polynY, k64f centerX, k64f centerY);


/**
*  Set lateral scaling of map pixel
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   scale       lateral scaling
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_SetXyScale(kS3dPolynomialMap polynMap, k64f scale);

/**
*  Set z resolution. The coefficient will be only used, if the input data is integer.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   scale       Z resolution.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_SetZscale(kS3dPolynomialMap polynMap, k64f scale);

/**
* Calculates the polynomial functions from two phase images.
*
* @public              @memberof kS3dPolynomialMap
* @param   polynMap    kS3dPolynomialMap object.
* @param   data        Input map of kPoint3d32f.
* @param   polynMCX    Output polynomial function: phaseX(camX, camY).
* @param   polynMCY    Output polynomial function: phaseY(camX, camY).
* @param   polynCMX    Output polynomial inverse function: CamX(phaseX, phaseY).
* @param   polynCMY    Output polynomial inverse function: CamY(phaseX, phaseY).
* @param   picInfo     Handle of kS3dPictureInfoItem structure.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomialMap_BidirectionPolynMapFromXY(kS3dPolynomialMap polynMap, kArray2 data,
    kS3dPolynomial polynMCX, kS3dPolynomial polynMCY, kS3dPolynomial polynCMX, kS3dPolynomial polynCMY, kS3dPictureInfoItem *picInfo);


#include <kVision/S3d/kS3dPolynomialMap.x.h>

#endif
