/**
* @file    kS3dPolynomial.h
* @brief   Declares the kS3dPolynomial class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_S3D_POLYN2D_H
#define KVISION_S3D_POLYN2D_H

#include <kApi/kApiDef.h>
#include <kApi/kAlloc.h>
#include <kVision/Common/kVision.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kArray3.h>
#include <kApi/Data/kArrayList.h>
#include <kApi/Data/kBytes.h>
#include <kApi/Data/kString.h>
#include <kVision/L3d/kL3dTransform3d.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <math.h>
#include <time.h>

#define kS3DPOLYNOMIALMAX_ORDER   (8)
#define kS3DPOLYNOMIAL8_COUNT     (45)

/**
* @class       kS3dPolynomial
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Provides some utility functions for polynomial object. The polynomial order is limited to between 1 and 8.
*/
typedef kObject kS3dPolynomial;

/**
* Constructs a kS3dPolynomial object
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_Construct(kS3dPolynomial* polynomial, kAlloc allocator);

/**
* Get the polynomial value at x and y: z(x,y) = sum (a * x^i * y ^j)
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @param   x           coordinate x.
* @param   y           coordinate y.
* @return              polynomial value at x and y.
*/
kVsFx(k64f) kS3dPolynomial_Value(kS3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the polynomial value at x and y with coefficient a(0) == 0.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @param   x           coordinate x.
* @param   y           coordinate y.
* @return              polynomial value at x and y.
*/
kVsFx(k64f) kS3dPolynomial_ValueZero(kS3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the partial derivative value of x at x and y: dz(x,y) / dx.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @param   x           coordinate x. 
* @param   y           coordinate y.
* @return              Partial derivative value.
*/
kVsFx(k64f) kS3dPolynomial_DxValue(kS3dPolynomial polynomial, k64f x, k64f y);

/**
* Get the first coefficient a(0) of the polynomial.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @return              first coefficient.
*/
kVsFx(k64f) kS3dPolynomial_Offset(kS3dPolynomial polynomial);

/**
* Set the first coefficient a(0) of the polynomial.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @param   offset      Value to be set for the first coefficient a(0)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_SetOffset(kS3dPolynomial polynomial, k64f offset);

/**
* Get the order of the polynomial.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @return              Order of the polynomial.
*/
kVsFx(k16s) kS3dPolynomial_Order(kS3dPolynomial polynomial);

/**
* Set the order of the polynomial.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object
* @param   order       Order of the polynomial
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_SetOrder(kS3dPolynomial polynomial, k16s order);


/**
* rescale the coefficients.
*
* @public              @memberof kS3dPolynomial.
* @param   polynomial  kS3dPolynomial object
* @param   factor      Scaling factor.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_Scaling(kS3dPolynomial polynomial, k64f factor);

/**
* Get the coefficient of the polynomial at the given position: a(index).
*
* @public              @memberof kS3dPolynomial.
* @param   polynomial  kS3dPolynomial object.
* @param   Index       position index.
* @return              coefficient of the polynomial at the given position.
*/
kVsFx(k64f) kS3dPolynomial_CoefficientItem(kS3dPolynomial polynomial, k32s Index);

/**
* Set all coefficient of the polynomial to zero.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  Destination for the constructed object handle.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_Zero(kS3dPolynomial polynomial);

/**
* Checking whether all coefficient of the polynomial are zero.
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object.
* @return              Operation status.
*/
kVsFx(kBool) kS3dPolynomial_IS_NULL(kS3dPolynomial polynomial);

/**
* Copy contents of a polynomial
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  Destination kS3dPolynomial object. 
* @param   source      Source kS3dPolynomial object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_Assign(kS3dPolynomial polynomial, kS3dPolynomial source);

/**
* Get polymimal coefficents
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object.
* @return              polymimal coefficents.
*/
kVsFx(k64f*) kS3dPolynomial_Coefficients(kS3dPolynomial polynomial);

/**
* Replace all the coefficients in the class
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object.
* @param   coeffs      Handle of k64f array. All the coefficients in the class will be replaced by the array.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_SetCoefficients(kS3dPolynomial polynomial, const k64f* coeffs);

/**
* Add all coefficient of two polynomials
*
* @public              @memberof kS3dPolynomial
* @param   outPolyn    kS3dPolynomial object. outPolyn += addPolyn
* @param   addPolyn    To be added polynomial.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_Add(kS3dPolynomial outPolyn, kS3dPolynomial addPolyn);

/**
* Add a coefficient at given position
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object. a[Index] += offset
* @param   Index       Coefficient position.  
* @param   offset      To be added value.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_AddItem(kS3dPolynomial polynomial, k32s Index, k64f offset);

#ifdef  KS3DVOLUMECHECK_DEBUGLOG

/**
* Writes the coefficients into a log file only used for debug
*
* @public              @memberof kS3dPolynomial
* @param   polynomial  kS3dPolynomial object.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPolynomial_DebugValues(kS3dPolynomial polynomial);

#endif 

#include <kVision/S3d/kS3dPolynomial.x.h>

#endif /* KVISION_S3D_POLYN2D_H */
