/**
* @file    kS3dPhaseProcessor.x.h
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_PROCESSOR_X_H
#define kS3D_PHASE_PROCESSOR_X_H

#define kS3D_PHASE_PROCESSOR_DEFAULT_INTENSITY_TYPE      (kS3D_STRIPE_INTENSITY_COMBINE)
#define kS3D_PHASE_PROCESSOR_VIEW_COUNT                  (2)

#define kS3D_PHASE_PROCESSOR_UNWRAPPED_PHASE_BIT_DEPTH   (17)  // unwrapped phase bit depth aka [base code + phase]
#define kS3D_PHASE_PROCESSOR_BASE_CODE_BIT_DEPTH         (8)   // stripe binary or EPS base bit depth

#define kS3D_PHASE_PROCESSOR_MIN_OUTLIER_RUN             (8)
#define kS3D_PHASE_PROCESSOR_OUTLIER_STEP_THRESHOLD      (0.02)  // as a fraction of the full Z range

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dPhaseProcessorExecutionTime
{
    kSize timeRangeFilter;
    kSize timeLookup;
    kSize timeIntensity;
    kSize timeCudaSync;
} kS3dPhaseProcessorExecutionTime;

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

typedef struct kS3dPhaseProcessorClass
{
    kObjectClass base;
    kS3dStereoProfiler profiler;

    // Compression ///////////////////////////////////////////////////////
    kBool compressionEnabled;
    kSize compressionSubframeCount;

    kSize imageWidth;
    kSize imageHeight;

    // Stripe ////////////////////////////////////////////////////////////
    kSize referenceImageCount;
    kSize stripeImageCount;
    kSize phaseImageCount;

    // EPS ///////////////////////////////////////////////////////////////
    kArray1 embeddedPhaseCoefficients; // k64f
    kArray1 embeddedPhaseImageCounts;  // kSize

    kS3dEncodeType encodeType; // Stripe, EPS

    // Other params //////////////////////////////////////////////////////
    kSize parallelCount;
    kBool imageInputEnabled;
    kBool rangeFilterEnabled;

    kSize sequenceCount;
    kSize intensitySequenceIndex;

    kS3dStripeIntensityType intensityType;

    kSize phaseContrastThreshold;
    kSize stripeContrastThreshold;

    kSize phaseBeginSample;
    kSize phasePeriodSampleCount;
    kSize outputHeight;

    k64f ySampleBegin;
    k64f ySampleStep;
    kSize outputWidth;

    kS3dPhaseCorrectionType phaseCorrectionType;
    kSize erosionMargin;

    // Internal data /////////////////////////////////////////////////////
    kSize phasePeriod;
    kSize phaseShift;  // phaseStep = 2^phaseShift = 1 << phaseShift
    kSize phaseBeginScaled;
    kSSize ySampleBeginScaled;
    kSize ySampleStepScaled;

    kVsJobQueue jobQueue;

    // Transpose support /////////////////////////////////////////////////
    kBool transposeEnabled;
    kArray2 transposedPhasePixel2[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kImage transposedImage[kS3D_PHASE_PROCESSOR_VIEW_COUNT];

    // Algs //////////////////////////////////////////////////////////////
    kS3dPhaseViewProcessor viewProcessor[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kS3dPhaseViewDecompressor viewDecompressor[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kS3dRangeFilter rangeFilter;
    kS3dSurfaceGenerator surfaceGenerator;

    // Buffers ///////////////////////////////////////////////////////////
    kArray2 viewDecompressorPhasePixel2[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray2 viewXProj[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray2 viewIntensity[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray1 phaseTable;

    // Execution timing //////////////////////////////////////////////////
    kTimer timer;
    kS3dPhaseProcessorExecutionTime executionTime;

    // Output buffers ////////////////////////////////////////////////////
    kArray2 outputIntensity;
    kArray2 outputRanges;

    // CUDA support //////////////////////////////////////////////////////
    kCudaStream cudaViewStream[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray2 cudaViewXProj[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray2 cudaOutputRanges;
    kArray2 hostOutputRanges; // for now we fall back to HOST if kS3D_STRIPE_LOOKUP_MERGE_QUADRATIC or rangeFilterEnabled, deprecate once we have full Cuda support 

    kArray2 cudaViewIntensity[kS3D_PHASE_PROCESSOR_VIEW_COUNT];
    kArray2 cudaOutputIntensity;

    kArray1 cudaPhaseTable;

    kBool retainInput;
    kBool cudaEnabled;
    kBool cudaManagedMemoryEnabled;
} kS3dPhaseProcessorClass;

kDeclareClassEx(kVs, kS3dPhaseProcessor, kObject)

//semi-private exported functions (virtual override methods)

kVsFx(kStatus) kS3dPhaseProcessor_VInitClone(kS3dPhaseProcessor processor, kS3dPhaseProcessor source, kAlloc allocator);
kVsFx(kStatus) kS3dPhaseProcessor_VRelease(kS3dPhaseProcessor processor);
kVsFx(kSize)   kS3dPhaseProcessor_VSize(kS3dPhaseProcessor processor);

//semi-private exported functions for reporting/advanced usage(API can change at any time)
kVsFx(kSize) kS3dPhaseProcessor_ExecutionTime(kS3dPhaseProcessor processor);

///@param outputIntensity a kArray2<kPoint32s> of texture coordinates
kVsFx(kStatus) kS3dPhaseProcessor_BeginEx(kS3dPhaseProcessor processor, kArray2 outputRanges, kArray2 outputIntensity, kArray2 outputTexture0, kArray2 outputTexture1);
kVsFx(kS3dStereoProfiler) kS3dPhaseProcessor_Profiler(kS3dPhaseProcessor processor);

//non-exported (private) methods
kStatus kS3dPhaseProcessor_Init(kS3dPhaseProcessor processor, kS3dStereoProfiler profiler, kAlloc allocator);

kStatus kS3dPhaseProcessor_SetupTranspose(kS3dPhaseProcessor processor);
kStatus kS3dPhaseProcessor_SetupDecompressor(kS3dPhaseProcessor processor);
kStatus kS3dPhaseProcessor_SetupPhaseTable(kS3dPhaseProcessor processor);

kStatus kS3dPhaseProcessor_CombineIntensity(kS3dPhaseProcessor processor, kArray2 intensity0, kArray2 intensity1, kArray2 output);
#if defined (K_HAVE_CUDA)
kStatus kS3dPhaseProcessor_CombineIntensityCuda(kS3dPhaseProcessor processor, kArray2 intensity0, kArray2 intensity1, kArray2 output);
#else
kInline kStatus kS3dPhaseProcessor_CombineIntensityCuda(kS3dPhaseProcessor processor, kArray2 intensity0, kArray2 intensity1, kArray2 output) { return kERROR; }
#endif

kBool kS3dPhaseProcessor_IsManagedMemoryEnabled();

kStatus kS3dPhaseProcessor_StartTimer(kS3dPhaseProcessor processor);
kSize kS3dPhaseProcessor_Elapsed(kS3dPhaseProcessor processor);

kStatus kS3dPhaseProcessor_End(kS3dPhaseProcessor processor);
kStatus kS3dPhaseProcessor_Lookup(kS3dPhaseProcessor processor);

//////////////////////////////////////////////////////////////////////////
// Inline
//
// PhasePeriod = PhaseStep * PhasePeriodSampleCount
// PhaseStep   = 2^PhaseShift
//////////////////////////////////////////////////////////////////////////

kInlineFx(kSize) kS3dPhaseProcessor_CalculatePhaseShift(kSize phasePeriodSampleCount)
{
    const kSSize phaseBitDepth = kS3D_PHASE_PROCESSOR_UNWRAPPED_PHASE_BIT_DEPTH - kS3D_PHASE_PROCESSOR_BASE_CODE_BIT_DEPTH;
    const kSSize phaseShift = phaseBitDepth - kMath_Log2Ceil32u((k32u)phasePeriodSampleCount);

    if (phaseShift < 0) return kSIZE_NULL;

    return (kSize)phaseShift;
}

kInlineFx(kSize) kS3dPhaseProcessor_CalculatePhasePeriod(kSize phasePeriodSampleCount)
{
    kSize phaseShift = kS3dPhaseProcessor_CalculatePhaseShift(phasePeriodSampleCount);

    if (phaseShift == kSIZE_NULL) return kSIZE_NULL;

    return phasePeriodSampleCount * ((kSize)1 << phaseShift);
}

//////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////

#endif  /* #ifndef kS3D_PHASE_PROCESSOR_X_H */
