/**
* @file    kS3dPhaseFilter.h
* @brief   Declares the kS3dPhaseFilter class.
*
* @internal
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef kS3D_PHASE_FILTER_H
#define kS3D_PHASE_FILTER_H

#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>

/**
* @class       kS3dPhaseFilter
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Performs filtering operations on decoded phase and stripe maps
*/
typedef kObject kS3dPhaseFilter;

/**
* Constructs a kS3dPhaseFilter object
*
* @public              @memberof kS3dPhaseFilter
* @param   filter     Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_Construct(kS3dPhaseFilter* filter, kAlloc alloc);

/**
* Updates working buffers of the implementation to reflect current set of algorithm parameters.
* Calling this function is optional, as this validation step is also performed during each execution
* of the algorithm
* However, the initialization time may be non-negligible, which would affect the first execution of the algorithm.
* To eliminate the added delay from the first algorithm execution, the user should call kS3dPhaseFilter_Setup
* after all of the parameters have been configured.
*
* @public              @memberof kS3dPhaseFilter
* @param   filter      Filter object
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_Setup(kS3dPhaseFilter filter);

/**
* Fills small gaps in stripe code data and corrects stripe boundary errors.
*
* @public                   @memberof kS3dPhaseFilter
* @param   filter           Filter object
* @param   stripePhaseMap   Phase decoding map (kArray2<kPhasePixel2>). The data in the map is replaced with the
*                           filtered result
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_SnapCodeMap(kS3dPhaseFilter filter, kArray2 stripePhaseMap);

/**
* Erodes phase maps by removing specified width of valid phase data around valid boundary
*
* @public                   @memberof kS3dPhaseFilter
* @param   filter           Filter object
* @param   margin           Number of pixels around the valid phase map boundary to remove
* @param   stripePhaseMap   Phase decoding map (kArray2<kPhasePixel2>). The data in the map is replaced with the
*                           eroded results
* @return                   Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_Erode(kS3dPhaseFilter filter, kSize margin, kArray2 stripePhaseMap);

/**
* Sets the size of the expected input images.
*
* @public              @memberof kS3dPhaseFilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_SetImageSize(kS3dPhaseFilter filter, kSize imageWidth, kSize imageHeight);

/**
* Gets the size of input images
*
* @public              @memberof kS3dPhaseFilter
* @param   filter      Filter object
* @param   imageWidth  Width of the input images
* @param   imageHeight Height of the input images
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_ImageSize(kS3dPhaseFilter filter, kSize* imageWidth, kSize* imageHeight);

/**
* Sets the scale of the phase period. The resulting phase output is scaled such that 0 to 2Pi phase values are mapped to 0 to period
*
* @public              @memberof kS3dPhaseFilter
* @param   filter     Filter object
* @param   period      Phase period scale
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseFilter_SetPhasePeriod(kS3dPhaseFilter filter, kSize period);

/**
* Gets the scale of the phase period.
*
* @public              @memberof kS3dPhaseFilter
* @param   filter     Filter object
* @return              Phase period scale
*/
kVsFx(kSize) kS3dPhaseFilter_PhasePeriod(kS3dPhaseFilter filter);

#include <kVision/S3d/kS3dPhaseFilter.x.h>

#endif  /* kS3D_PHASE_FILTER_H */
