/**
* @file    kS3dPhaseBlockInfo.h
* @brief   Declares the kS3dPhaseBlockInfo class.
*
* @internal
* Copyright (C) 2014-2022 by LMI Technologies Inc. All rights reserved.
*/
#ifndef kS3D_PHASE_BLOCK_INFO_H
#define kS3D_PHASE_BLOCK_INFO_H

#include <kVision/S3d/kS3dCommon.h>
#include <kApi/Data/kXml.h>
#include <kVision/L3d/kL3dTransform3d.h>

#define kS3D_PHASE_BLOCK_NAME             "Phase"
#define kS3D_PHASE_BLOCK_TYPE_NAME        "Phase"

#define kS3D_PHASE_BLOCK_IN_IMAGE0_NAME   "InImage-0"
#define kS3D_PHASE_BLOCK_IN_IMAGE1_NAME   "InImage-1"

#define kS3D_PHASE_BLOCK_IN_PHASE0_NAME   "InPhase-0"
#define kS3D_PHASE_BLOCK_IN_PHASE1_NAME   "InPhase-1"

#define kS3D_PHASE_BLOCK_OUT_RANGES_NAME     "OutRanges"
#define kS3D_PHASE_BLOCK_OUT_INTENSITY_NAME  "OutIntensity"
#define kS3D_PHASE_BLOCK_OUT_DIAG_NAME       "OutDiag"
#define kS3D_PHASE_BLOCK_OUT_PLOT0_NAME      "OutPlot-0"
#define kS3D_PHASE_BLOCK_OUT_PLOT1_NAME      "OutPlot-1"

typedef enum
{
    kS3D_PHASE_BLOCK_IN_IMAGE0  = 0,
    kS3D_PHASE_BLOCK_IN_IMAGE1  = 1,
    kS3D_PHASE_BLOCK_IN_PHASE0  = 2,
    kS3D_PHASE_BLOCK_IN_PHASE1  = 3,

    kS3D_PHASE_BLOCK_OUT_RANGES = 101,
    kS3D_PHASE_BLOCK_OUT_INTENSITY = 102,
    kS3D_PHASE_BLOCK_OUT_DIAG = 201,
    kS3D_PHASE_BLOCK_OUT_PLOT0  = 202,
    kS3D_PHASE_BLOCK_OUT_PLOT1  = 203
} kS3dPhaseBlockPort;

/**
* @class   kS3dPhaseBlockInfo
* @ingroup kVision
* @brief   Settings class for kS3dPhaseBlock
*/
typedef kObject kS3dPhaseBlockInfo;

/**
* Constructs a kS3dPhaseBlockInfo object
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Construct(kS3dPhaseBlockInfo* info, kAlloc allocator);

/**
* Reads values from a kXml object
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object.
* @param   xml         Source kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Read(kS3dPhaseBlockInfo info, kXml xml, kXmlItem item);

/**
* Writes values to a kXml object
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object.
* @param   xml         destination kXml object
* @param   item        Root xml node
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Write(kS3dPhaseBlockInfo info, kXml xml, kXmlItem item);

/**
* Loads a new kS3dPhaseBlockInfo from a kXml object
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Destination for the constructed object handle.
* @param   xml         Source kXml object
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Load(kS3dPhaseBlockInfo* info, kXml xml, kAlloc allocator);

/**
* Saves a kS3dPhaseBlockInfo to a new kXml object
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Object to be saved
* @param   xml         Pointer to a destination kXml handle.
* @param   allocator   Memory allocator (or kNULL for default).
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Save(kS3dPhaseBlockInfo info, kXml* xml, kAlloc allocator);

/**
* Sets the name of the calibration file to be loaded from storage. 
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   name        Calibration file name
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetCalFileName(kS3dPhaseBlockInfo info, const kChar* name);

/**
* Gets the name of the calibration file to be loaded from storage.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Calibration file name
*/
kVsFx(const kChar*) kS3dPhaseBlockInfo_CalFileName(kS3dPhaseBlockInfo info);

/**
* Sets the numerical resolution (units) of X,Y,Z range data output. Must be sufficiently high to ensure
* that all post-transformation range coordinates fit within 16-bit range
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   resolution  Range resolution
* @return              Operation status
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetRangeResolution(kS3dPhaseBlockInfo info, k64f resolution);

/**
* Gets the numerical resolution (units) of X,Y,Z range data output.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Range resolution
*/
kVsFx(k64f) kS3dPhaseBlockInfo_RangeResolution(kS3dPhaseBlockInfo info);

/**
* Sets active area region of interest (applied pre-transform)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   roi         Active area region of interest
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetActiveAreaRoi(kS3dPhaseBlockInfo info, const kRect3d64f* roi);

/**
* Gets active area region of interest (applied pre-transform)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Active area region of interest
*/
kVsFx(const kRect3d64f*) kS3dPhaseBlockInfo_ActiveAreaRoi(kS3dPhaseBlockInfo info);

/**
* Sets profile data transformation
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   transform   Profile data transformation
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetTransform(kS3dPhaseBlockInfo info, const kL3dTransform3d* transform);

/**
* Gets profile data transformation
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Profile data transformation
*/
kVsFx(const kL3dTransform3d*) kS3dPhaseBlockInfo_Transform(kS3dPhaseBlockInfo info);

/**
* Sets Camera 0 window parameters
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   window      Camera 0 window parameters
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetWindow0(kS3dPhaseBlockInfo info, const kL3dCameraWindow* window);

/**
* Gets Camera 0 window parameters
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Camera 0 window parameters
*/
kVsFx(const kL3dCameraWindow*) kS3dPhaseBlockInfo_Window0(kS3dPhaseBlockInfo info);

/**
* Sets Camera 1 window parameters
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   window      Camera 1 window parameters
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetWindow1(kS3dPhaseBlockInfo info, const kL3dCameraWindow* window);

/**
* Gets Camera 1 window parameters
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Camera 1 window parameters
*/
kVsFx(const kL3dCameraWindow*) kS3dPhaseBlockInfo_Window1(kS3dPhaseBlockInfo info);

/**
* Sets range lookup type (kS3dStripeLookupType)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   lookupType  Range lookup type (kS3dStripeLookupType)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetLookupType(kS3dPhaseBlockInfo info, kSize lookupType);

/**
* Gets range lookup type (kS3dStripeLookupType)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Range lookup type (kS3dStripeLookupType)
*/
kVsFx(kSize) kS3dPhaseBlockInfo_LookupType(kS3dPhaseBlockInfo info);

/**
* Sets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   begin       Starting rectified Y value expressed in mm.
* @param   step        Step size along the rectified Y axis in mm (must be a multiple of kS3dStereoCal_Parameters(cal)->projectionResolution
*                      to ensure that the full expected ROI is generated
* @param   count       Number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetYProjectionRoi(kS3dPhaseBlockInfo info, k64f begin, k64f step, kSize count);

/**
* Gets the parameters determining resampling region of interest along the rectified Y axis (columns in both camera and projection images)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   begin       Output pointer to the starting rectified Y value (mm)
* @param   step        Output pointer to the step size along the rectified Y axis (mm)
* @param   count       Output pointer to the number of points along the rectified Y axis (output buffer width)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_YProjectionRoi(kS3dPhaseBlockInfo info, k64f* begin, k64f* step, kSize* count);

/**
* Sets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public                         @memberof kS3dPhaseBlockInfo
* @param   info                   Info object
* @param   periodSampleCount      Number of samples per phase repetition
* @param   beginSample            First sample position to output.  Calculated as (fractional stripe begin) * periodSampleCount
* @param   count                  Total number of projection samples. Calculated as (fractional stripe span) * periodSampleCount.
*                                 The value determines output range and intensity buffer height
* @return                         Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetPhaseRoi(kS3dPhaseBlockInfo info, kSize periodSampleCount, kSize beginSample, kSize count);

/**
* Gets the parameters determining resampling region of interest along projection phase modulation axis (rows in both camera and projection images)
*
* @public                         @memberof kS3dPhaseBlockInfo
* @param   info                   Info object
* @param   periodSampleCount      Output pointer to the number of samples per phase repetition
* @param   beginSample            Output pointer to the first sample position to output. Calculated as fractional stripe begin * periodSampleCount
* @param   count                  Output pointer to the total number of projection samples. Calculated as fractional stripe span * periodSampleCount.
*                                 The value determines output range and intensity buffer height
* @return                         Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_PhaseRoi(kS3dPhaseBlockInfo info, kSize* periodSampleCount, kSize* beginSample, kSize* count);

/**
* Sets the maximum number of threads allowed to be used for processing. Set to one for blocking single
* threaded operation
*
* Single threaded operation is configured as the default.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   count       Maximum number of threads allowed for processing.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetParallelCount(kS3dPhaseBlockInfo info, kSize count);

/**
* Gets the maximum number of threads allowed to be used for processing.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Maximum number of threads allowed for processing.
*/
kVsFx(kSize) kS3dPhaseBlockInfo_ParallelCount(kS3dPhaseBlockInfo info);

/**
* Enables image input. Depending on the setting the block exposes inputs for either
* image data (kS3D_PHASE_BLOCK_IN_IMAGEx) or stripe and phase map 
* data (kS3D_PHASE_BLOCK_IN_STRIPEx, kS3D_PHASE_BLOCK_IN_PHASEx)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   enable      Flag specifying whether image input should be enabled
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_EnableImageInput(kS3dPhaseBlockInfo info, kBool enable);

/**
* Gets the value of the flag specifying whether raw image input is enabled.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Flag specifying whether image input is enabled
*/
kVsFx(kBool) kS3dPhaseBlockInfo_ImageInputEnabled(kS3dPhaseBlockInfo info);

/**
* Enable cuda acceleration.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   enable      Flag specifying whether cuda should be enabled.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_EnableCuda(kS3dPhaseBlockInfo info, kBool enable);

/**
* Gets the value of the flag specifying whether cuda is enabled.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Flag specifying whether cuda is enabled
*/
kVsFx(kBool) kS3dPhaseBlockInfo_CudaEnabled(kS3dPhaseBlockInfo info);

/**
* Enable intensity processing.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   enable      Flag specifying whether intensity should be enabled.
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_EnableIntensity(kS3dPhaseBlockInfo info, kBool enable);

/**
* Gets the value of the flag specifying whether intensity is enabled.
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Flag specifying whether intensity is enabled
*/
kVsFx(kBool) kS3dPhaseBlockInfo_IntensityEnabled(kS3dPhaseBlockInfo info);

/**
* Sets the lengths of the Gray code and phase subsequences. At most one reference image is supported.
* If no stripe sequence is used (phase period occupies full projection)
* the number of reference images must also be zero. Total number of expected images is the sum of the individual
* sub-sequence counts
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @param   referenceImageCount   Number of reference images in the image sequence (<= 1)
* @param   stripeImageCount      Number of stripe images in the image sequence (<= 7)
* @param   phaseImageCount       Number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetStripeSequenceInfo(kS3dPhaseBlockInfo info, kSize referenceImageCount, kSize stripeImageCount, kSize phaseImageCount);

/**
* Gets the lengths of the Gray code and phase subsequences.
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @param   referenceImageCount   Output pointer to the number of reference images in the image sequence
* @param   stripeImageCount      Output pointer to the number of stripe images in the image sequence
* @param   phaseImageCount       Output pointer to the number of phase images in the sequence
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_StripeSequenceInfo(kS3dPhaseBlockInfo info, kSize* referenceImageCount, kSize* stripeImageCount, kSize* phaseImageCount);

/**
* Sets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                     @memberof kS3dPhaseBlockInfo
* @param   info               Info object
* @param   coefficients       Array of coefficients describing embedded periods (k64f).
* @param   stepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   coefficientCount   Size of the "coefficients" and "stepCounts" arrays.
* @return                     Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetEmbeddedPhaseSequenceInfo(kS3dPhaseBlockInfo info, const k64f* coefficients, const kSize* stepCounts, kSize coefficientCount);

/**
* Gets period coefficients and the number of images per period.
* Coefficients are used to calculate the phase periods per set of patterns used.
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @param   outCoefficients       Array of coefficients describing embedded periods (k64f).
* @param   outStepCounts         Array specifying number of images per period to use (kSize same size as "coefficients" and == coefficientCount).
* @param   outCoefficientCount   Number of values in the "coefficients" and "stepCounts" arrays returned (outCoefficientCount <= kS3D_EMBEDDED_PHASE_DECODER_PERIOD_COUNT).
* @param   capacity              Size of "coefficients" and "stepCounts" arrays passed in (capacity should be >= coefficientCount).
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_EmbeddedPhaseSequenceInfo(kS3dPhaseBlockInfo info, k64f* outCoefficients, kSize* outStepCounts, kSize capacity);

/**
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @return                        Number of coefficients.
*/

kVsFx(kSize) kS3dPhaseBlockInfo_EmbeddedPhaseCoefficientCount(kS3dPhaseBlockInfo info);


/**
* Sets the total number of decoding sequences.
*
* @public                @memberof kS3dPhaseBlockInfo
* @param   info          Info object
* @param   sequenceCount Number of expected decoding sequences (default is 1)
* @return                Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetSequenceCount(kS3dPhaseBlockInfo info, kSize count);

/**
* Gets the total number of decoding sequences
*
* @public                @memberof kS3dPhaseBlockInfo
* @param   info          Info object
* @return                Number of expected decoding sequences
*/
kVsFx(kSize) kS3dPhaseBlockInfo_SequenceCount(kS3dPhaseBlockInfo info);

/**
* Sets contrast threshold for stripe decoding. The difference between stripe image pixel intensity and reference pixel intensity
* must be higher or equal to this value in order for the decoding at that pixel to be valid. Note that the value set by this function
* has no effect if map input is used
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   threshold   Contrast threshold for stripe decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetStripeContrastThreshold(kS3dPhaseBlockInfo info, kSize threshold);

/**
* Gets contrast threshold for stripe decoding
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Contrast threshold for stripe decoding
*/
kVsFx(kSize) kS3dPhaseBlockInfo_StripeContrastThreshold(kS3dPhaseBlockInfo info);

/**
* Sets contrast threshold for phase decoding. The range of intensities for a given pixel location among all phase
* images must exceed this threshold value in order for the phase decoding to produce a valid result. Note that the 
* value set by this function has no effect if map input is used
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   threshold   Contrast threshold for phase decoding
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetPhaseContrastThreshold(kS3dPhaseBlockInfo info, kSize threshold);

/**
* Gets contrast threshold for phase decoding
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Contrast threshold for phase decoding
*/
kVsFx(kSize) kS3dPhaseBlockInfo_PhaseContrastThreshold(kS3dPhaseBlockInfo info);

/**
* Sets encode type (kS3dEncodeType)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @param   encodeType  Encode type (kS3dEncodeType)
* @return              Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetEncodeType(kS3dPhaseBlockInfo info, kS3dEncodeType encodeType);

/**
* Gets encode type (kS3dEncodeType)
*
* @public              @memberof kS3dPhaseBlockInfo
* @param   info        Info object
* @return              Encode type (kS3dEncodeType)
*/
kVsFx(kS3dEncodeType) kS3dPhaseBlockInfo_EncodeType(kS3dPhaseBlockInfo info);

/**
* Sets phase decoder format (kCameraPhaseDecoderFormat)
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @param   format                Format (kCameraPhaseDecoderFormat)
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetPhaseDecoderFormat(kS3dPhaseBlockInfo info, kCameraPhaseDecoderFormat format);

/**
* Gets phase decoder format (kCameraPhaseDecoderFormat)
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @return                        Format (kCameraPhaseDecoderFormat)
*/
kVsFx(kCameraPhaseDecoderFormat) kS3dPhaseBlockInfo_PhaseDecoderFormat(kS3dPhaseBlockInfo info);

/**
* Sets the number of subframes to split the phase for compression (DecoderFormat == kCAMERA_PHASE_DECODER_FORMAT_DELTA_I8_P24 == 8).
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @param   subframeCount         Subframe count
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_SetSubframeCount(kS3dPhaseBlockInfo info, kSize subframeCount);

/**
* Gets phase prediction.
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  Info object
* @return                        Subframe count
*/
kVsFx(kSize) kS3dPhaseBlockInfo_SubframeCount(kS3dPhaseBlockInfo info);

/**
* Set all parameters of kS3dPhaseBlockInfo object to default values
*
* @public                        @memberof kS3dPhaseBlockInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Clear(kS3dPhaseBlockInfo info);

/**
* Validate the kS3dPhaseBlockInfo object passed in
*
* @public                        @memberof kS3dPhaseActionInfo
* @param   info                  kS3dPhaseActionInfo object
* @return                        Operation status.
*/
kVsFx(kStatus) kS3dPhaseBlockInfo_Validate(kS3dPhaseBlockInfo info);



#include <kVision/S3d/kS3dPhaseBlockInfo.x.h>

#endif
