/**
 * @file     kMp3dProfiler.h
 * @brief    Declares the kMp3dProfiler class
 * 
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */

#ifndef K_VISION_MP3D_PROFILER_H
#define K_VISION_MP3D_PROFILER_H

#include <kVision/Mp3d/kMp3dCommon.h>
#include <kVision/L3d/kL3dCommon.h>
#include <kVision/L3d/kL3dPolynomial2Fit.h>
#include <kVision/L3d/kL3dPolynomial2Array.h>
#include <kFireSync/Data/kSpot.h>
#include <kVision/Mp3d/kMp3dSensorCal.h>
#include <kVision/L3d/kL3dTransform2d.h>

/**
 * kMp3dProfiler max view count
 *
 * @relates        kMp3dProfiler
 */
#define kMP3D_PROFILER_MAX_VIEW_COUNT                       (4)

/**
 * @class       kMp3dProfiler
 * @extends     kObject
 * @ingroup     kVision-Mp3d
 * @brief       Used in creating and using LUT for x and z ranges.
 */
typedef kObject kMp3dProfiler;

/**
 * Constructs a kMp3dProfiler object.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  cal         Calibration data used to generate LUT. Must live for the duration of the profiler object.
 * @param  allocator   Memory allocator. 
 * @return             Operation status
 */
kVsFx(kStatus) kMp3dProfiler_Construct(kMp3dProfiler* profiler, kMp3dSensorCal cal, kAlloc allocator);

/**
 * Generates the LUT based on the calibration data. This function needs to be called before any other LUT functions can be used.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             Operation status
 */
kVsFx(kStatus) kMp3dProfiler_Setup(kMp3dProfiler profiler);

/**
* Applies the new settings to the LUT.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @param  profileRect Active area region of interest. X is for X axis, Y is for Z axis. Use kNULL for full sensor FOV. This
*                      should be in the sensor's frame of reference prior to any transformations.
* @param  transform   X and Z transformation applied to LUT. Use kNULL if no transform is needed.
* @return             Operation status
*/
kVsFx(kStatus) kMp3dProfiler_Refresh(kMp3dProfiler profiler, const kRect64f* profileRect, const kL3dTransform2d* transform);

/**
* Calculates the optimum X and Z resolutions after applying an active area and input transform to the LUTs.
*
* @public               @memberof kMp3dProfiler
* @param  profiler      kMp3dProfiler object.
* @param  profileRect   Active area region of interest. X is for X axis, Y is for Z axis. Use kNULL for full sensor FOV. This
*                       should be in the sensor's frame of reference prior to any transformations.
* @param  transform     X and Z transformation applied to LUT. Use kNULL if no transform is needed.
* @param  xResolution   Output X resolution.
* @param  zResolution   Output Z resolution.
* @return               Operation status
*/
kVsFx(kStatus) kMp3dProfiler_CalculateResolutions(kMp3dProfiler profiler, const kRect64f* profileRect, const kL3dTransform2d* transform, k64f* xResolution, k64f* zResolution);

/**
* Calculates the camera window to produce a desired active area. Active area X and width are not supported due to the PL alg requiring
* spot-free columns on the left of the image for background noise estimation. The calculated camera window includes margins so that the
* full spot is not clipped at the window edges (required for tracheid performance).
*
* @public               @memberof kMp3dProfiler
* @param  profiler      kMp3dProfiler object.
* @param  profileRect   Active area region of interest. X is for X axis, Y is for Z axis. This should be in the sensor's
*                       frame of reference prior to any transformations.
* @param  viewIndex     Camera index.
* @param  window        Output camera window.
* @return               Operation status
*/
kVsFx(kStatus) kMp3dProfiler_ActiveArea(kMp3dProfiler profiler, const kRect64f* profileRect, kSize viewIndex, kL3dCameraWindow* window);

/**
 * Access the LUT.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  viewIndex   Index of view.
 * @return             Operation status
 */
kVsFx(kArray2) kMp3dProfiler_Lut(kMp3dProfiler profiler, kSize viewIndex);

/**
 * Gets the max allowed spots with the same ID.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             Max allowed spots with the same ID.
 */
kVsFx(kSize) kMp3dProfiler_MaxIdDuplicateCount(kMp3dProfiler profiler);

/**
 * Sets the max allowed spots with the same ID.
 * 
 * @public                      @memberof kMp3dProfiler
 * @param  profiler             kMp3dProfiler object.
 * @param  maxIdDuplicateCount  Max allowed spots with the same ID.
 * @return                      Operation status
 */
kVsFx(kStatus) kMp3dProfiler_SetMaxIdDuplicateCount(kMp3dProfiler profiler, kSize maxIdDuplicateCount);

/**
 * Gets the X resolution of the LUT.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             X resolution.
 */
kVsFx(k64f) kMp3dProfiler_XResolution(kMp3dProfiler profiler);

/**
* Sets the X resolution of the LUT.
*
* @public               @memberof kMp3dProfiler
* @param  profiler      kMp3dProfiler object.
* @param  resolution    X resolution.
* @return               Operation status
*/
kVsFx(kStatus) kMp3dProfiler_SetXResolution(kMp3dProfiler profiler, k64f resolution);

/**
 * Gets the Z resolution of the LUT.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             Z resolution.
 */
kVsFx(k64f) kMp3dProfiler_ZResolution(kMp3dProfiler profiler);

/**
* Sets the Z resolution of the LUT.
*
* @public               @memberof kMp3dProfiler
* @param  profiler      kMp3dProfiler object.
* @param  resolution    Z resolution.
* @return               Operation status
*/
kVsFx(kStatus) kMp3dProfiler_SetZResolution(kMp3dProfiler profiler, k64f resolution);

/**
 * Gets the run window.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  viewIndex   Index of view.
 * @return             Run window.
 */
kVsFx(kL3dCameraWindow) kMp3dProfiler_RunWindow(kMp3dProfiler profiler, kSize viewIndex);

/**
 * Sets the run window.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  runWindow   Run window.
 * @param  viewIndex   Index of view.
 * @return             Operation status
 */
kVsFx(kStatus) kMp3dProfiler_SetRunWindow(kMp3dProfiler profiler, kL3dCameraWindow runWindow, kSize viewIndex);

/**
* Gets the field of view.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @return             Field of view.
*/
kVsFx(const kRect3d64f*) kMp3dProfiler_Fov(kMp3dProfiler profiler);

/**
* Sets the field of view.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @param  fov         Field of view.
* @return             Operation status
*/
kVsFx(kStatus) kMp3dProfiler_SetFov(kMp3dProfiler profiler, const kRect3d64f* fov);

/**
* Sets the imager granularity. The granularity is used in active area calculations. If no granularity is set, the default is 1.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @param  y           Y granularity
* @param  height      Height granularity
* @param  width       Width granularity
* @return             Operation status
*/
kVsFx(kStatus) kMp3dProfiler_SetGranularity(kMp3dProfiler profiler, k32u y, k32u height, k32u width);

/**
* Gets the imager Y granularity.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @return             Y granularity
*/
kVsFx(k32u) kMp3dProfiler_YGranularity(kMp3dProfiler profiler);

/**
* Gets the imager height granularity.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @return             Height granularity
*/
kVsFx(k32u) kMp3dProfiler_HeightGranularity(kMp3dProfiler profiler);

/**
* Gets the imager width granularity.
*
* @public             @memberof kMp3dProfiler
* @param  profiler    kMp3dProfiler object.
* @return             Width granularity
*/
kVsFx(k32u) kMp3dProfiler_WidthGranularity(kMp3dProfiler profiler);

/**
 * Gets the reject weak multiples factor. Prior to range lookup, spots are sorted and filtered. Spots with
 * the same ID are filtered out if weak or are merged together. For all spots with the same ID, spots whose
 * strength is less than the max strength by this factor are removed.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             The reject weak multiples factor value.
 */
kVsFx(k32s) kMp3dProfiler_RejectWeakMultiplesFactor(kMp3dProfiler profiler);

/**
 * Sets the reject weak multiples factor value. Prior to range lookup, spots are sorted and filtered. Spots with
 * the same ID are filtered out if weak or are merged together. For all spots with the same ID, spots whose
 * strength is less than the max strength by this factor are removed.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  factor      The reject weak multiples factor value.
 * @return             Operation status
 */
kVsFx(kStatus) kMp3dProfiler_SetRejectWeakMultiplesFactor(kMp3dProfiler profiler, k32s rejectWeakMultiplesFactor);

/**
 * Gets the ID correction minimum separation value. For all spots with the same ID, spots whose centroid column
 * difference is greater than this minimum are split into neighbouring spots. The remaining spots with the same
 * ID are merged.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @return             The ID correction minimum separation value. In pixels.
 */
kVsFx(k32u) kMp3dProfiler_IdCorrectionMinSeparation(kMp3dProfiler profiler);

/**
 * Sets the ID correction minimum separation value. For all spots with the same id, spots whose centroid column
 * difference is greater than this minimum are split into neighbouring spots. The remaining spots with the same
 * ID are merged.
 * 
 * @public                              @memberof kMp3dProfiler
 * @param  profiler                     kMp3dProfiler object.
 * @param  idCorrectionMinSeperation    The ID correction minimum separation value. In pixels.
 * @return                              Operation status
 */
kVsFx(kStatus) kMp3dProfiler_SetIdCorrectionMinSeparation(kMp3dProfiler profiler, k32u idCorrectionMinSeperation);

/**
 * Gets the value to specify if the coordinates should be flipped.
 * 
 * @public                              @memberof kMp3dProfiler
 * @param  profiler                     kMp3dProfiler object.
 * @return                              Flip the coordinates?.
 */
kVsFx(kBool) kMp3dProfiler_CoordinatesFlipped(kMp3dProfiler profiler);

/**
 * Sets the value to specify if the coordinates should be flipped.
 * 
 * @public                              @memberof kMp3dProfiler
 * @param  profiler                     kMp3dProfiler object.
 * @param  flipCoordinates              Flip the coordinates?.
 * @return                              Operation status
 */
kVsFx(kStatus) kMp3dProfiler_FlipCoordinates(kMp3dProfiler profiler, kBool flipCoordinates);

/**
 * Process the spots to determine their coordinates.
 * 
 * @public                  @memberof kMp3dProfiler
 * @param  profiler         kMp3dProfiler object.
 * @param  viewIndex        Index of view.
 * @param  temperature      Internal sensor temperature (in degrees).
 * @param  spots            Input spots.
 * @param  count            Input spots count.
 * @param  outRanges        Output ranges.
 * @param  outSortedSpots   Output sorted spots.
 * @param  outCount         Output count.
 * @return                  Operation status
 */
kVsFx(kStatus) kMp3dProfiler_Run(kMp3dProfiler profiler, kSize viewIndex, k64f temperature, const kSpot2* spots, kSize count, kPoint16s* outRanges, kSpot2* outSortedSpots, kSize outCount);

/**
 * Access the real world X data used in creating the LUT.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  viewIndex   Index of view.
 * @return             X data array.
 */
kVsFx(kArray2) kMp3dProfiler_XLut(kMp3dProfiler profiler, kSize viewIndex);

/**
 * Access the ID data.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  viewIndex   Index of view.
 * @return             ID data array.
 */
kVsFx(kArray2) kMp3dProfiler_IdLut(kMp3dProfiler profiler, kSize viewIndex);

/**
 * Access the real world Z data used in creating the LUT.
 * 
 * @public             @memberof kMp3dProfiler
 * @param  profiler    kMp3dProfiler object.
 * @param  viewIndex   Index of view.
 * @return             Z data array.
 */
kVsFx(kArray2) kMp3dProfiler_ZLut(kMp3dProfiler profiler, kSize viewIndex);


/** @relates  kMp3dProfiler @{ */

#define kMP3D_PROFILER_INTERP(V0, V1, FRACT)                                                                kxMP3D_PROFILER_INTERP(V0, V1, FRACT)
#define kMP3D_PROFILER_INTERP_RANGE_LIN(P0, P1, FRACT, OUT)                                                 kxMP3D_PROFILER_INTERP_RANGE_LIN(P0, P1, FRACT, OUT)
#define kMP3D_PROFILER_RANGE_LOOKUP(PROFILER, VIEWINDEX, SPOTINDEX, TEMPINDEX, X, Y, XWORLD, ZWORLD)        kxMP3D_PROFILER_RANGE_LOOKUP(PROFILER, VIEWINDEX, SPOTINDEX, TEMPINDEX, X, Y, XWORLD, ZWORLD)

/** @} */

#include <kVision/Mp3d/kMp3dProfiler.x.h>

#endif /* #ifndef K_VISION_MP3D_PROFILER_H */

