/**
 * @file kMp3dGeoCal.h
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
 */

#ifndef K_VISION_MP3D_GEO_CAL_H
#define K_VISION_MP3D_GEO_CAL_H

#include <kFireSync/kFsDef.h>
#include <kVision/Common/kVision.h>

/**
 * Calibration file name.
 *
 * @relates kMp3dGeoCal
 */
#define kMP3D_GEO_CAL_FILE_NAME    "GeoCal.xml"

/**
 * @class       kMp3dGeoCal
 * @extends     kObject
 * @ingroup     kVision-Mp3d
 * @brief       Vision calibration data. This class stores camera geometric orientation information to allow for pitch, yaw, 
                roll, and X and Y translation correction, and it stores resolution information over the range for resampling color 
                images in the field. This class contains the following information per camera: The imager X center (in mm) as a 
                function of Z (mm), Y center as a function of Z, X resolution (mm/pixel) as a function of Z, Y resolution as 
                a function of Z, and the roll angle (degrees) of the imager. Typically, color cameras require all four fits, whereas
                profile cameras do not require X and Y resolution over Z. Fits are stored as polynomial coefficients; These can be 
                applied using the kMp3dGeoCalUtils class.

 */
typedef kObject kMp3dGeoCal;

/**
 * Constructs a kMp3dGeoCal object.
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @param  cameraCount         Count of cameras.
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dGeoCal_Construct(kMp3dGeoCal* cal, kSize cameraCount, kAlloc allocator);

/**
 * Load Geo cal data.
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @param  node                Node to load file from. Use kNULL to load from disk.
 * @param  fileName            Filename.
 * @param  allocator           Memory allocator.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dGeoCal_Load(kMp3dGeoCal* cal, kNode node, const kChar* fileName, kAlloc allocator);

/**
 * Save Geo cal data.
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @param  node                Node to load file from. Use kNULL to save to disk.
 * @param  fileName            Filename.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dGeoCal_Save(kMp3dGeoCal cal, kNode node, const kChar* filename);

/**
 * Gets sensor ID
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @return                     Sensor ID.
 */
kVsFx(k32u) kMp3dGeoCal_Id(kMp3dGeoCal cal);

/**
 * Sets sensor ID
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @param  id                  Sensor ID.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dGeoCal_SetId(kMp3dGeoCal cal, k32u id);

/**
 * Gets timestamp
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @return                     Timestamp.
 */
kVsFx(const kChar*) kMp3dGeoCal_Timestamp(kMp3dGeoCal cal);

/**
 * Sets timestamp
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @param  timestamp           Timestamp.
 * @return                     Operation status
 */
kVsFx(kStatus) kMp3dGeoCal_SetTimestamp(kMp3dGeoCal cal, const kChar* timestamp);

/**
 * Gets count of cameras.
 * 
 * @public                     @memberof kMp3dGeoCal
 * @param  cal                 kMp3dGeoCal object.
 * @return                     Count of views.
 */
kVsFx(kSize) kMp3dGeoCal_CameraCount(kMp3dGeoCal cal);

/**
* Gets calibration window.
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  x                   Output parameter for window x. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @param  y                   Output parameter for window y. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @param  width               Output parameter for window width. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @param  height              Output parameter for window height. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @param  xSubsampling        Output parameter for window xSubsampling. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @param  ySubsampling        Output parameter for window ySubsampling. Provide with a pointer to a k32u. Ignored if input is kNULL.
* @return                     Operation status.
*/
kVsFx(kStatus) kMp3dGeoCal_CalWindow(kMp3dGeoCal cal, kSize cameraIndex, k32u* x, k32u* y, k32u* width, k32u* height, k32u* xSubsampling, k32u* ySubsampling);

/**
* Sets calibration window.
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  x                   Window x.
* @param  y                   Window y.
* @param  width               Window width. 
* @param  height              Window height.
* @param  xSubsampling        Window xSubsampling.
* @param  ySubsampling        Window ySubsampling.
* @return                     Operation status.
*/
kVsFx(kStatus) kMp3dGeoCal_SetCalWindow(kMp3dGeoCal cal, kSize cameraIndex, k32u x, k32u y, k32u width, k32u height, k32u xSubsampling, k32u ySubsampling);

/**
* Sets X resolution to Z fit. 
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images, 
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output. 
* The fit is stored as polynomial coefficients. The X resolution at a given Z range (mm) can be determined by:
* 
*       X resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  coefficients        Array of fit coefficients. Type: k64f.
* @return                     Operation status
*/
kVsFx(kStatus) kMp3dGeoCal_SetXResolutionFit(kMp3dGeoCal cal, kSize cameraIndex, kArray1 coefficients);

/**
* Gets X resolution to Z fit.
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images,
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output.
* Profile cameras would have a fit set to kNULL. The fit is stored as polynomial coefficients. The X resolution at a 
* given Z range (mm) can be determined by:
*
*       X resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @return                     Array of X resolution to Z fit coefficients. Type: k64f. kNULL is returned if the 
*                             camera does not have a X resolution to Z fit.
*/
kVsFx(kArray1) kMp3dGeoCal_XResolutionFit(kMp3dGeoCal cal, kSize cameraIndex);

/**
* Sets Y resolution to Z fit.
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images,
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output.
* The fit is stored as polynomial coefficients. The Y resolution at a given Z range (mm) can be determined by:
*
*       Y resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  coefficients        Array of Y resolution to Z fit coefficients. Type: k64f.
* @return                     Operation status
*/
kVsFx(kStatus) kMp3dGeoCal_SetYResolutionFit(kMp3dGeoCal cal, kSize cameraIndex, kArray1 coefficients);

/**
* Gets Y resolution to Z fit.
* The X and Y resolution (mm/pixel) of an imager's pixels varies with the range. When resampling images,
* it is necessary to know the native resolution of each pixel. Typically, this is only needed for color/texture output.
* Profile cameras would have a fit set to kNULL. The fit is stored as polynomial coefficients. The Y resolution at a
* given Z range (mm) can be determined by:
*
*       Y resolution = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @return                     Array of Y resolution to Z fit coefficients. Type: k64f. kNULL is returned if the
*                             camera does not have a X resolution to Z fit.
*/
kVsFx(kArray1) kMp3dGeoCal_YResolutionFit(kMp3dGeoCal cal, kSize cameraIndex);

/**
* Sets X center to Z fit.
* The world X center (mm) of an imager varies with the range if any yaw is present. It also represents the distance
* from the sensor reference 0 plane (usually mid-sensor). The fit is stored as polynomial coefficients. The X center
* at a given Z range (mm) can be determined by:
*
*       X center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  coefficients        Array of X center to Z fit coefficients. Type: k64f.
* @return                     Operation status
*/
kVsFx(kStatus) kMp3dGeoCal_SetXCenterFit(kMp3dGeoCal cal, kSize cameraIndex, kArray1 coefficients);

/**
* Gets X center to Z fit.
* The world X center (mm) of an imager varies with the range if any yaw is present. It also represents the distance
* from the sensor reference 0 plane (usually mid-sensor). The fit is stored as polynomial coefficients. The X center
* at a given Z range (mm) can be determined by:
*
*       X center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @return                     Array of X center to Z fit coefficients. Type: k64f.
*/
kVsFx(kArray1) kMp3dGeoCal_XCenterFit(kMp3dGeoCal cal, kSize cameraIndex);

/**
* Sets Y center to Z fit.
* The world Y center (mm) of an imager varies with the range if any pitch is present. It also represents the distance
* from the sensor reference 0 plane. The fit is stored as polynomial coefficients. The Y center at a given Z range (mm) 
* can be determined by:
*
*       Y center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @param  coefficients        Array of Y center to Z fit coefficients. Type: k64f.
* @return                     Operation status
*/
kVsFx(kStatus) kMp3dGeoCal_SetYCenterFit(kMp3dGeoCal cal, kSize cameraIndex, kArray1 coefficients);

/**
* Gets Y center to Z fit.
* The world Y center (mm) of an imager varies with the range if any pitch is present. It also represents the distance
* from the sensor reference 0 plane (usually mid-sensor). The fit is stored as polynomial coefficients. The Y center
* at a given Z range (mm) can be determined by:
*
*       Y center = coefficients[0] + coefficients[1]*Z + ... + coefficients[N]*Z^N
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index. kERROR_PARAMETER if invalid.
* @return                     Array of Y center to Z fit coefficients. Type: k64f.
*/
kVsFx(kArray1) kMp3dGeoCal_YCenterFit(kMp3dGeoCal cal, kSize cameraIndex);

/**
* Sets camera roll angle (radians). This can be used to correct for camera roll.
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index
* @param  angle               Roll angle (in radians)
* @return                     Operation status
*/
kVsFx(kStatus) kMp3dGeoCal_SetRoll(kMp3dGeoCal cal, kSize cameraIndex, k64f roll);

/**
* Gets camera roll angle. This can be used to correct for camera roll.
*
* @public                     @memberof kMp3dGeoCal
* @param  cal                 kMp3dGeoCal object.
* @param  cameraIndex         Camera index
* @return                     Roll angle (in radians)
*/
kVsFx(k64f) kMp3dGeoCal_Roll(kMp3dGeoCal cal, kSize cameraIndex);

#include <kVision/Mp3d/kMp3dGeoCal.x.h>

#endif /* #ifndef K_VISION_MP3D_GEO_CAL_H */

