#ifndef K_VISION_L3D_LUTWRITERLEGACYIMPL_H
#define K_VISION_L3D_LUTWRITERLEGACYIMPL_H

#include <kVision/L3d/kL3dSensorCal.h>
#include <kVision/L3d/kL3dTransform2d.h>
#include <kFireSync/Client/Camera/kCameraInfo.h>
#include <kFireSync/Client/kCamera.h>

#include <kVision/L3d/kL3dPolynomialCalData.h>
#include <kVision/L3d/kL3dPolynomialArray.h>
#include <kVision/L3d/kL3dChartPlotter.h>
#include <kVision/L3d/kL3dUtilities.h>
#include <kVision/Common/kNullMath.h>
#include <kFireSync/Data/kSpot.h>
#include <kFireSync/Utils/kPlRangeLut.h>
#include <kApi/Data/kArray2.h>
#include <kApi/Data/kMath.h>
#include <kVision/Common/kMatrix.h>

typedef kObject kL3dLutWriterLegacyImpl;

typedef struct kL3dLutWriterLegacyImplRunParams 
{
    k64f xResolution;
    k64f zResolution;

    k32u intensityMultiplier;
} kL3dLutWriterLegacyImplRunParams;

/**
 * @brief Construct a Legacy Lut Writer Implementation object
 * 
 * @param writer    Handler to lut object
 * @param camera    Camera object to get image dimensions and subsampling intervals
 * @param cal       Calibration file to generate luts
 * @param caps      if camera is kNULL kRangeLutCaps can be passed in to specify default camera parameters
 * @param allocator create object with a specific memory allocator (e.g. unit tests)
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_Construct(kL3dLutWriterLegacyImpl* writer, kCamera camera, kL3dSensorCal cal, kRangeLutCaps* caps, kAlloc allocator);

/**
 * @brief Get the window subdivision used in the lut
 * 
 * @param writer    Handler to lut object
 * @return k32u     Value of the window subdivision used in the lut
 */
kVsFx(k32u)    kL3dLutWriterLegacyImpl_WindowSubdivision(kL3dLutWriterLegacyImpl writer);

/**
 * @brief Given an FOV or active area and transform, determine the x and z resolution (mm/k16s)
 * 
 * @param writer        Handler to lut object
 * @param roi           Rect object that describes the FOV or active area desired
 * @param transform     Transform to compensate sensor orientations
 * @param xResolution   Pointer to output x resolution
 * @param zResolution   Pointer to output z resolution
 * @return kStatus      kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_CalculateResolutions(kL3dLutWriterLegacyImpl writer, const kRect64f* roi, const kL3dTransform2d* transform, k64f* xResolution, k64f* zResolution);

/**
 * @brief Given the runtime parameters (FOV and intensity multiplier) and transform, recreate the look up table and write to the FPGA
 * 
 * @param writer            Handler to lut object
 * @param transform         Transform to compensate sensor orientations
 * @param params            Runtime parameters (Resolutions and intensity multiplier)
 * @param exportSettings    Flag to enable lut settings export
 * @return kStatus 
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_Refresh(kL3dLutWriterLegacyImpl writer, const kL3dTransform2d* transform, const kL3dLutWriterLegacyImplRunParams* params, kBool exportSettings);

/**
 * @brief Getter functions that return the calibration tables at various stages of processing
 * 
 * Cal: Dense table without extrapolation
 * Array: Subsampled Cal table with extrapolation
 * Lut: Array table scaled to k16s
 * Valid: Table that flags which entires in the table are valid
 */
kVsFx(kArray2) kL3dLutWriterLegacyImpl_XCal(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_ZCal(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_XLut(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_ZLut(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_ValidLut(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_XArray(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_ZArray(kL3dLutWriterLegacyImpl writer);
kVsFx(kArray2) kL3dLutWriterLegacyImpl_ValidArray(kL3dLutWriterLegacyImpl writer);

/**
 * @brief Given a centre and slice (x,y) test whether a valid range look up can be performed
 * 
 * @param writer    Handler to lut object
 * @param slice     spot slice value (y)
 * @param centre    spot centre value (x)
 * @return kBool    kTRUE if valid, kFALSE if failed or invalid
 */
kVsFx(kBool) kL3dLutWriterLegacyImpl_ValidateCentroid(kL3dLutWriterLegacyImpl writer, k32u slice, k64f centre);

/**
 * @brief Given spot slice and centre, perform FPGARL and get world space x and z values
 * 
 * @param writer    Handler to lut object
 * @param slice     spot slice value (y)
 * @param centre    spot centre value (x)
 * @param xOutput   Output: World x 
 * @param zOutput   Output: World z
 * @return kBool    kTRUE if valid, kFALSE if failed or invalid
 */
kVsFx(kBool) kL3dLutWriterLegacyImpl_GetRanges(kL3dLutWriterLegacyImpl writer, k32u slice, k64f centre, k64f* xOutput, k64f* zOutput);

/**
 * @brief Writes out LUT parameters used (Step Size, Intensity, Lut Size) to local sensor storage as a kdat6 file
 * 
 * @param writer        Handler to lut object
 * @param params        Pointer to output the camera default parameters
 * @param xArray        Pointer to output x LUT table
 * @param zArray        Pointer to output z LUT table
 * @param validArray    Pointer to output valid table
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_ExportSettings(kL3dLutWriterLegacyImpl writer, const kRangeLutParams* params, const kRangeLutArray* xArray,
    const kRangeLutArray* zArray, const kRangeLutArray* validArray);

/**
 * @brief Writes out look up tables as csv and kdat6 to local sensor storage
 * 
 * @param writer    Handler to lut object
 * @param storage   Storage object from kNode
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_ExportLuts(kL3dLutWriterLegacyImpl writer, kStorage storage);

/**
 * @brief Save the lut plot to local root directory as kdat6 file
 * 
 * @param writer    Handler to lut object
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_ExportLutPlot(kL3dLutWriterLegacyImpl writer);

/**
 * @brief Create and return lut plot as kPlot object
 * 
 * This class does not own the kPlot object and expects the display class to destroy it
 * 
 * @param writer    Handler to lut object
 * @param output    pointer to returned kPlot
 * @return kStatus  kOK if successful, kERROR otherwise
 */
kVsFx(kStatus) kL3dLutWriterLegacyImpl_LutPlot(kL3dLutWriterLegacyImpl writer, kPlot* output);

#include <kVision/L3d/kL3dLutWriterLegacyImpl.x.h>

#endif /* #ifndef K_VISION_L3D_LUTWRITERLEGACYIMPL_H */
