/**
* @file    kG3dContourMatching.h
* @brief   Declares the kG3dContourMatching class. 
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef KVISION_kG3D_CONTOURMATCHING_H
#define KVISION_kG3D_CONTOURMATCHING_H
#include <kApi/Threads/kTimer.h>
#include <kVision/Common/kVision.h>
#include <kVision/S3d/kS3dCommon.h>
#include <kVision/G3d/kG3dJumpContour.h>
#include <kVision/G3d/kG3dContourTemplate.h>
#include <kVision/G3d/kG3dAffineMergeHeightMap.h>
#include <kVision/G3d/kG3dSegmentPlanes.h>
#include <kVision/S3d/kS3dVolumeCheckUtilities.h>
#include <kVision/G3d/kG3dPalette.h>

/**
* @class       kG3dContourMatching
* @extends     kObject
* @ingroup     kVision-S3d
* @brief       Main class for measurement of magnet glue band height.
*
*              This class is desined for the measurement of magnet glue band height 
*              with the two known types.
*              The main work steps of the calss are as following:
*
*              1.  The image data is smoothed by a Gaussian function;
*              2.  The image is differentiated with respect to the directions x and y;  
*              3.  After the gradient has been calculated at each point of the image, 
*                  the edges can be located at the points of local maximum gradient magnitude;
*              4.  The edge direction is also calculated for each pixel, whose gradient magnitude 
*                  is larger than a certain threshold, which provides also important information for
*                  the contour detection;
*              5.  The points of contour candidate is segmented to some groups;
*              6.  Four straight lines are detected from boundary contours by using Hough transform
*              7.  The left and right long side lines are calculated by using the Least squares best fit method;
*              8.  The top and bottom short side lines are calculated by using the Least squares best fit method;
*              9.  Four corner points are calculated according to specified calculation model. 
*                  The model depends on the straightness of the line, the length and location of the straight line, 
*                  and the reliability of the line detection;
*              10. The projected points from the weighted gaps center are calculated if they exist.
*              11. The points are sorted according to the defined coordinate system.
*              12. Calculating transform coefficients for the both maps.
*              13. Resampling texture map to the world coordinate system
*              14. Resampling height map to the world coordinate system
*              15. Modifying the actual glue border contours inside and outside separately according to the template 
*                  contour and the corresponding search direction.
*              16. Determining the intensity threshold within the glue band area with a histogram curve for detection 
*                  the overflow in magnet area and in cage area;
*              17. Creating a segmentation map to separate the glue area, magnet area and cage area;
*              18. Calculating the reference height for each track based on the segmentation map
*              19. Calculating all height parameters (output 1 - 5 ) for each track based on the segmentation map
*              20. Calculating all 2d parameters (output 6 - 11)  for each track based on modified glue band contours.
*              21. Check the average intensity within the area is lower than the threshold;
*              21. Check the average height within the area is higher than the height of surrounding area. 
                   If the conditions are met, the area can be identified as overflow.
*/
typedef kObject kG3dContourMatching;

/**
* Constructs a kG3dContourMatching object
*
* @public                  @memberof kG3dContourMatching
* @param   matching        Destination for the constructed object handle.
* @param   allocator       Memory allocator (or kNULL for default).
* @return                  Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_Construct(kG3dContourMatching* matching, kAlloc allocator);

/**
* Set the scaling coeffizients of the main height map.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   scale          scaling coeffizients of the main height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightMapScale(kG3dContourMatching matching, kPoint3d64f scale);

/**
* Set the offset coeffizients of the main height map.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   offset         Offset coeffizients of the main height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightMapOffset(kG3dContourMatching matching, kPoint3d64f offset);

/**
* Performs the evaluation of glue band height measurements without template 
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   heightMap      Data buffer for input height map.   kArray2 < k16s >
* @param   textureImg     Data buffer for input exteral bitmap   kImage < k8u > or kImage < kArgb > 
* @param   intensityMap   Data buffer for input intensity map.  kArray2 < k8u >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_RunBitmap(kG3dContourMatching matching, kArray2 heightMap, kImage textureImg, kArray2 intensityMap);

/**
* Performs the evaluation of glue band height measurements using template
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   heightMap      Data buffer for input height map.   kArray2 < k16s >
* @param   textureImg     Data buffer for input exteral bitmap   kImage < k8u > or kImage < kArgb >
* @param   intensityMap   Data buffer for input intensity map.  kArray2 < k8u >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_RunTemplate(kG3dContourMatching matching, kArray2 heightMap, kImage textureImg, kArray2 intensityMap);

/**
* Set optional height map from the buddy sensor
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   map            Data buffer for input buddy height map.   kArray2 < k16s >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightMap2(kG3dContourMatching matching, kArray2 map);

/**
* Set the optional scaling coeffizients of the buddy height map.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   scale          scaling coeffizients of the main height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightMapScale2(kG3dContourMatching matching, kPoint3d64f scale);

/**
* Set the optional offset coeffizients of the buddy height map.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   offset         Offset coeffizients of the buddy height map in the 3 directions.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightMapOffset2(kG3dContourMatching matching, kPoint3d64f offset);

/**
* Set optional intensity map from the buddy sensor
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   map            Data buffer for input buddy intensity map.  kArray2 < k8u >
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetIntensityMap2(kG3dContourMatching matching, kArray2 map);

/**
* Get the data buffer of interesting points, which are used for calculating the affine 
* transformation matrix, for a given image index.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   index          Index of the image series defined in kG3dInputImagesOrder
* @return                 Data buffer of interesting points for given image index.
*/
kVsFx(kArray1) kG3dContourMatching_MeasurementPoints(kG3dContourMatching matching, k32s index);

/**
* Get the data buffer of magnet track object, which includes information of glue band geometry for each track.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Data buffer of magnet track object.
*/
kVsFx(kArrayList) kG3dContourMatching_MeasurementTracks(kG3dContourMatching matching);

/**
* Get the data buffer of raw glue boundary contour, which is created as intermediate results
* when creating template. This function is used for visualizing the contours to check the 
* corretness of contour detection in debug mode.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Data buffer of raw glue boundary contour.
*/
kVsFx(kArrayList) kG3dContourMatching_InterProfiles(kG3dContourMatching matching);

/**
* Get a affine transformation matrix with a given index, which indicates the relationship between the images.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   mode           An index which indicates the relationship between the images
* @return                 kG3dContourTemplate object.
*/
kVsFx(kL3dTransform3d*) kG3dContourMatching_TransformMatrix(kG3dContourMatching matching, kG3dAffineTransformRelationMode mode);

/**
* Set cage thick. This parameter is used for estimate of the maximum range of cage surface 
* outside the glue band, which should be used for the evaluation of the reference plane each track. 
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   thick          Cage thick.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetFrameThick(kG3dContourMatching matching, k64f thick);

/**
* Get output data buffer of height map in the world coordinate system
* This function is used for visualizing the height map for kFactory app.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Output data buffer of height map.
*/
kVsFx(kArray2) kG3dContourMatching_OutputHeightMap(kG3dContourMatching matching);

/**
* Get output data buffer of segment map in the world coordinate system
* This function is used for visualizing the height map for kFactory app.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Output data buffer of segment map.
*/
kVsFx(kArray2) kG3dContourMatching_OutputMaskMap(kG3dContourMatching matching);

/**
* Set type of to be measured sample for kFactory app.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   mode           Type of to be measured sample: 0 = small magnet 1 = big magnet 2 = unknown
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetMode(kG3dContourMatching matching, kSize mode);

/**
* Get output data buffer of the external bitmap in the world coordinate system
* This function is used for visualizing the height map for kFactory app.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Output data buffer of segment map.
*/
kVsFx(kArray2) kG3dContourMatching_OutputBitmap(kG3dContourMatching matching);

/**
* Get output data buffer of the buddy intensity map in the world coordinate system
* This function is used for visualizing the height map for kFactory app.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Output data buffer of segment map.
*/
kVsFx(kArray2) kG3dContourMatching_OutputBuddyIntensityMap(kG3dContourMatching matching);

/**
* Change the dafault mode of creating raw data for affine transform matrix. 
* If doIt is set kTRUE, Instead of height map, the intensity map should be 
* used for creating interesting points.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   doIt           Intensity map should be used?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetTransformBuddyIntensity(kG3dContourMatching matching, kBool doIt);

/**
* Change the dafault domain mode of the affine transformation.
* If doIt is set kTRUE, Instead of height map, the external bitmap should be
* used as reference for the affinity correction.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   doIt           External bitmap should be used as reference?
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetTransformBitmap(kG3dContourMatching matching, kBool doIt);

/**
* Set the input data buffer for the lateral correlation measurements.
* This array includes the information about coordinate of each points measured by CMM.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   points         Input data buffer
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetCorrelationXYPoints(kG3dContourMatching matching, kArrayList points);

/**
* Set the input data buffer for the Z correlation measurements.
* This array includes the information about coordinate of each points measured by CMM.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   points         Input data buffer
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetCorrelationZPoints(kG3dContourMatching matching, kArrayList points);

/**
* Set the input data buffer for the lateral correlation measurements. 
* This array includes the information about lateral shift for each points
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   points         Input data buffer
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetCorrelationXYDifference(kG3dContourMatching matching, kArrayList points);

/**
* Get a kG3dContourTemplate object.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 kG3dContourTemplate object.
*/
kVsFx(kG3dContourTemplate) kG3dContourMatching_Template(kG3dContourMatching matching);

/**
* Get total running time
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Total running time.
*/
kVsFx(k64s) kG3dContourMatching_TotalTime(kG3dContourMatching matching);

/**
* Set sample index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   index          Sample index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetSampleIndex(kG3dContourMatching matching, kSize index);

/**
* Set repeat index of a measurement series.This function is only used for debug purpose to store
* the intermediate results without overwriting.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   index          Repeat index of a measurement series
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetRepeatIndex(kG3dContourMatching matching, kSize index);

/**
* Store a kG3dContourTemplate object into a file.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       file name for the template object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_StoreTemplate(kG3dContourMatching matching, const kChar* fileName);

/**
* Load a kG3dContourTemplate object from a file.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       file name for the template object.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_LoadTemplate(kG3dContourMatching matching, const kChar* fileName);

/**
* Return a message if all specified glue bands have been correctly detected.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 All specified glue bands have been correctly detected?
*/
kVsFx(kBool) kG3dContourMatching_IsTrackComplete(kG3dContourMatching matching);

/**
* Get the maximum limit of the height map for visualization
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Maximum limit of the height map
*/
kVsFx(k64f) kG3dContourMatching_MinimumZ(kG3dContourMatching matching);

/**
* Get the minimum limit of the height map for visualization
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 Minimum limit of the height map
*/
kVsFx(k64f) kG3dContourMatching_MaximumZ(kG3dContourMatching matching);

/**
* Get a output data buffer for the height measurements at defined position on glue bands
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   mask           Template object for evaluation of magnet glue bands
* @return                 Output data buffer for height measurements at defined position on glue bands
*/
kVsFx(kArrayList) kG3dContourMatching_CorrelationOutputZ(kG3dContourMatching matching);

/**
* Set a template object which includes the information about the evaluation of magnet glue bands 
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   mask           Template object for evaluation of magnet glue bands
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetTemplate(kG3dContourMatching matching, kG3dContourTemplate mask);

/**
* Set the output file name for storing a height map of remaining difference 
* between the two heightmaps after the alignment to each other
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing a difference height map.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetDifferenceFileName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing a external bitmap in the world coordinate system.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing a external bitmap in the world coordinate system.
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetTextureBmpName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing intensity map from the main sensor in the world coordinate system.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing intensity map from the main sensor
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetMainIntensityName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing intensity map from the buddy sensor in the world coordinate system.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing intensity map from the buddy sensor
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetBuddyIntensityName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing a color coded bitmap to illustrate overflow and gaps area.
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing a color coded bitamp
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetOverflowBmpName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing a color coded bitmap from the height map
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing a color coded bitmap from the height map
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightBmpName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing the height map
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing the height map
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightmapOutName(kG3dContourMatching matching, kChar* fileName);

/**
* Set the output file name for storing the aligned height map
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @param   fileName       Output file name for storing the aligned height map 
* @return                 Operation status.
*/
kVsFx(kStatus) kG3dContourMatching_SetHeightAlignOutName(kG3dContourMatching matching, kChar* fileName);

/**
* Get a color palette object for visualizing of a height map
*
* @public                 @memberof kG3dContourMatching
* @param   matching       kG3dContourMatching object.
* @return                 color palette object.
*/
kVsFx(kG3dPalette) kG3dContourMatching_Palette(kG3dContourMatching matching);

#include <kVision/G3d/kG3dContourMatching.x.h>

#endif  /* KVISION_kG3D_CONTOURMATCHING_H */
