/** 
 * @file    kFsLib.h
 *
 * @internal
 * Copyright (C) 2012-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LIB_H
#define K_FIRESYNC_LIB_H

#include <kFireSync/kFsDef.h>

/** 
 * Constructs the FireSync API type assembly. 
 *
 * This function initializes the FireSync API library and returns a handle that represents the library. 
 * When the library is no longer needed, pass the library handle to the kObject_Destroy function. 
 * 
 * Multiple nested calls to this function will return the same library instance. To ensure final cleanup, the 
 * kObject_Destroy function should be invoked a corresponding number of times on the library handle. 
 *
 * @param   assembly    Receives a handle to the kFsLib type assembly.
 * @return              Operation status. 
 */
kFsFx(kStatus) kFsLib_Construct(kAssembly* assembly);

/** 
 * Enables automatic creation of a global health provider for the process. 
 *
 * By default, a global health provider is not created for applications. This function 
 * can be used to override the default behaviour.  If enabled, a kHostHealth instance will 
 * be installed as the global health provider during kFsLib assembly initialization. 
 * 
 * This function can only be called <em>prior</em> to instantiating the kFsLib assembly. 
 * 
 * @param   enable    kTRUE to enable a default health provider. 
 * @return           
 */
kFsFx(void) kFsLib_EnableDefaultHealthProvider(kBool enable);

/** 
 * Sets the global health provider. 
 * 
 * If this function will be used to install a health provider, it should ideally be called at 
 * application startup time, just after initializing the kFireSync library but before any health 
 * or code profiling probes are constructed. 
 * 
 * It is the caller's responsibility to manage the lifetime of the provider instance; the kFsLib 
 * assembly will not destroy the provider on exit.  If a provider is registered, it should be 
 * unregistered prior to its destruction.
 * 
 * In sensor environments, a custom health provider is installed by the sensor framework. Sensor 
 * applications should not attempt to use kFsLib_SetHealthProvider; doing so may result in 
 * undefined behavior.
 * 
 * @param   provider    Health provider instance. 
 */
kFsFx(kStatus) kFsLib_SetHealthProvider(kHealth provider);

/** 
 * Gets the global health provider, if it exists. 
 * 
 * @return  Global health provider, or kNULL if not present. 
 */
kFsFx(kHealth) kFsLib_HealthProvider();

#include <kFireSync/kFsLib.x.h>

#endif
