/** 
 * @file    kHeapAlloc.h
 * @brief   Declares the kHeapAlloc class. 
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.
 */
#ifndef K_FIRESYNC_HEAP_ALLOC_H
#define K_FIRESYNC_HEAP_ALLOC_H

#include <kFireSync/kFsDef.h>

/**
 * @class   kHeapAlloc
 * @extends kAlloc
 * @ingroup kFireSync-Utils
 * @brief   Allocates memory from user-defined memory regions.
 *
 * This memory allocator is useful when the pre-allocated memory regions are needed to use for
 * dynamic memory allocations. kAlloc_Get/kAlloc_Free functions shall be used for allocations.
 *
 * Allocated blocks can all be freed at once by discarding the entire kHeapAlloc object without
 * calling kAlloc_Free function for each block separately.
 *
 * All public functions are thread-safe. kAlloc_Get/kAlloc_Free functions have O(log n) time complexity,
 * where n is number of free blocks in regions.
 */
// typedef kObject kHeapAlloc; --forward - declared in kFsDef.x.h

/**
 * Signature of callback to notify about heap exhaustion.
 *
 * @public                     @memberof kHeapAlloc
 * @param   heapAlloc          kHeapAlloc object.
 * @param   receiver           Pointer to the receiver specified in kHeapAlloc_AddExhaustionHandler function.
 * @param   minimumRegionSize  The smallest region that could be added to satisfy the memory request.
 * @return                     If callback returns kOK, kHeapAlloc will re-attempt allocation; otherwise,
 *                             the allocation operation will be allowed to fail.
 */

typedef kStatus(kCall* kHeapAllocExhaustionFx)(kHeapAlloc heapAlloc, kPointer receiver, kSize minimumRegionSize);

/**
 * @public  
 * @struct  kHeapAllocStats
 * @ingroup kFireSync-Utils
 * @brief   Structure for statistics calculated by kHeapAlloc_Stats()-function.
 */
struct kHeapAllocStats
{
    kSize memoryRegionsCount;   ///< Number of memory regions added.  

    kSize allocatedBlocksCount; ///< Number of memory blocks allocated.
    kSize allocatedBytesCount;  ///< Number of bytes in all allocated memory blocks.

    kSize freeBlocksCount;      ///< Number of free blocks available for allocations.
    kSize freeBytesCount;       ///< Number of bytes in all free blocks.

    kSize maxFreeBlockSize;     ///< The largest free block available for allocation.
};

/** 
 * Constructs a new kHeapAlloc instance. 
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   Receives the constructed kHeapAlloc instance. 
 * @param   allocator   Memory allocator for this object instance (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kHeapAlloc_Construct(kHeapAlloc* heapAlloc, kAlloc allocator);

/**
 * Adds a pre-allocated memory region for use of the kHeapAlloc-object.
 *
 * At least one memory block must be added for each kHeapAlloc instance before any allocation requests.
 * Additional memory regions can be added any time.
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   kHeapAlloc object.
 * @param   address     Start address of the pre-allocated memory region.
 * @param   size        Block size, in bytes.
 * @return              Operation status.
 */
kFsFx(kStatus) kHeapAlloc_AddMemoryRegion(kHeapAlloc heapAlloc, kPointer address, kSize size);

/**
 * Removes all added memory regions including allocated blocks.
 *
 * No need to call this function if kHeapAlloc object is destructed.
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   kHeapAlloc object.
 * @return              Operation status.
 */
kFsFx(kStatus) kHeapAlloc_RemoveMemoryRegions(kHeapAlloc heapAlloc);

/**
 * Removes all allocated blocks but leaves the existing memory regions in place.
 *
 * No need to call this function if kHeapAlloc object is destructed.
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   kHeapAlloc object.
 * @return              Operation status.
 */
kFsFx(kStatus) kHeapAlloc_Clear(kHeapAlloc heapAlloc);

/**
 * Reports used and free blocks from all added memory regions.
 *
 * The function call has O(log n) time complexity, where n is number of free blocks.
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   kHeapAlloc object.
 * @param   stats       Calculated statistics.
 * @return              Operation status.
 */
kFsFx(kStatus) kHeapAlloc_Stats(kHeapAlloc heapAlloc, kHeapAllocStats* stats);

/**
 * Adds a callback that will be invoked upon memory exhaustion.
 *
 * Various internal locks may be held by the thread that invokes the callback; accordingly, the implementation of the callback
 * should avoid acquiring any additional locks that might result in deadlock.
 *
 * @public              @memberof kHeapAlloc
 * @param   heapAlloc   kHeapAlloc object.
 * @param   callback    Callback to notify about heap exhaustion.
 * @param   receiver    Pointer to the received which is passed to the callback.
 * @return              Operation status.
 */
kFsFx(kStatus) kHeapAlloc_AddExhaustionHandler(kHeapAlloc heapAlloc, kHeapAllocExhaustionFx callback, kPointer receiver);

#include <kFireSync/Utils/kHeapAlloc.x.h>

#endif
