/** 
 * @file    kElf.h
 * @brief   ELF definitions.
 *
 * @internal
 * Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ELF_H
#define K_FIRESYNC_ELF_H

#include <kApi/kApiDef.h>

/* 
 * References:
 * https://en.wikipedia.org/wiki/Executable_and_Linkable_Format
 * http://refspecs.linuxbase.org/elf/elf.pdf
 */

#define kELF_ENDIAN_LITTLE              (1)
#define kELF_ENDIAN_BIG                 (2)

#define kELF_CLASS_32                   (1)
#define kELF_CLASS_64                   (2)

#define kELF_MACHINE_ARM                (0x28)

#define kELF_SEGMENT_TYPE_LOAD          (1)

#define kELF_PERMISION_EXECUTE          (0x01)

#define kELF_SECTION_TYPE_PROGBITS      (1)
#define kELF_SECTION_TYPE_SYMTAB        (2)
#define kELF_SECTION_TYPE_STRTAB        (3)

typedef k32u kElf32Address;

typedef struct kElf32Identity
{
    kChar magic[4];                     //0x7F followed by ELF in ASCII; these four bytes constitute the magic number.
    k8u archClass;                      //This byte is set to either 1 or 2 to signify 32- or 64-bit format, respectively.
    k8u endian;                         //This byte is set to either 1 or 2 to signify little or big endianness, respectively. This affects interpretation of multi-byte fields starting with offset 0x10.
    k8u elfVersion;                     //Set to 1 for the original version of ELF.
    k8u osAbi;                          //Identifies the target operating system ABI; It is often set to 0 regardless of the target platform.
    k8u osAbiVersion;                   //Further specifies the ABI version. Its interpretation depends on the target ABI. Linux kernel (after at least 2.6) has no definition of it.[4] In that case, offset and size of EI_PAD are 8.
    k8u reserved[7];                    //Currently unused
} kElf32Identity; 

typedef struct kElf32Header
{
    kElf32Identity identity;        
    k16u moduleType;                    //1, 2, 3, 4 specify whether the object is relocatable, executable, shared, or core, respectively.
    k16u machine;                       //Specifies target instruction set architecture. Some examples are: 0x02 SPARC, 0x03 x86, 0x28 ARM, 0x32 IA-64, 0x3E x86-64, 0xB7 AArch64. 
    k32u moduleVersion;                 //Set to 1 for the original version of ELF.
    kElf32Address entryPoint;           //This is the memory address of the entry point from where the process starts executing. This field is either 32 or 64 bits long depending on the format defined earlier.
    k32u programHeaderOffset;           //Points to the start of the program header table. It usually follows the file header immediately making the offset 0x40 for 64-bit ELF executables.
    k32u sectionHeaderOffset;           //Points to the start of the section header table.
    k32u flags;                         //Interpretation of this field depends on the target architecture.
    k16u headerSize;                    //Contains the size of this header, normally 64 bytes for 64-bit and 52 for 32-bit format.
    k16u programEntrySize;              //Contains the size of a program header table entry.
    k16u programEntryCount;             //Contains the number of entries in the program header table.
    k16u sectionEntrySize;              //Contains the size of a section header table entry.
    k16u sectionEntryCount;             //Contains the number of entries in the section header table.
    k16u nameSectionIndex;              //Contains index of the section header table entry that contains the section names.
} kElf32Header; 

typedef struct kElf32SectionHeader
{
    k32u name;                          //This member gives the section name (as index into string table)
    k32u type;                          //This member gives the section type
    k32u flags;                         //This member gives the section flags
    kElf32Address address;              //This member gives the section virtual addr at execution
    k32u offset;                        //This member gives the section file offset
    k32u size;                          //This member gives the section size in bytes
    k32u link;                          //This member gives the link to another section
    k32u info;                          //Additional section information
    k32u addralign;                     //This member gives the section alignment
    k32u entsize;                       //Entry size if section holds table
} kElf32SectionHeader; 

typedef struct kElf32Symbol
{
    k32u name;                      //This member gives the name (as index into string table)
    kElf32Address value;            //This member gives the symbol value
    k32u size;                      //This member gives the symbol size
    kByte info;                     //This member gives the type and binding
    kByte other;                    //This member has no meaning
    k16u shndx;                     //This member gives the section header index
} kElf32Symbol;

typedef struct kElf32ProgramEntry
{
    k32u type;                      //This member tells what kind of segment this array element describes or how to interpret the array element's information. 
    k32u offset;                    //This member gives the offset from the beginning of the file at which the first byte of the segment resides.
    kElf32Address virtAddr;         //This member gives the virtual address at which the first byte of the segment resides in memory.
    kElf32Address physAddr;         //On systems for which physical addressing is relevant, this member is reserved for the segment's physical address. This member requires operating system specific information, which is described in the appendix at the end of Book III.
    k32u fileSize;                  //This member gives the number of bytes in the file image of the segment; it may be zero. 
    k32u memSize;                   //This member gives the number of bytes in the memory image of the segment; it may be zero.
    k32u flags;                     //This member gives flags relevant to the segment. Defined flag values appear below.
    k32u align;                     //Loadable process segments must have congruent values for p_vaddr and p_offset, modulo the page size.This member gives the value to which the segments are aligned in memory and in the file. 
                                    //Values 0 and 1 mean that no alignment is required. Otherwise, p_align should be a positive, integral power of 2, and p_addr should equal p_offset, modulo p_align. 
} kElf32ProgramEntry;

#endif
