/** 
 * @file    kPxBlock.h
 * @brief   Declares the kPxBlock class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_BLOCK_H
#define K_FIRESYNC_PX_BLOCK_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kPxBlock
 * @extends kObject
 * @ingroup kFireSync-Pipe
 * @brief   Abstract base class for pipe block implementations. 
 */
//typedef kObject kPxBlock;         --forward-declared in kFsDef.x.h

/** 
 * Given a kPxBlock subclass type, gets a description of the class. 
 * 
 * This method should be overridden by subclasses to provide a description of each block type.
 *
 * @public              @memberof kPxBlock
 * @param   type        Block type. 
 * @return              Type description. 
 */
 kFsFx(const kChar*) kPxBlock_Description(kType type);

/** 
 * Factory constructor for block objects. 
 * 
 * This function should be used to construct all block types. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Receives constructed block object. 
 * @param   type        Block class type. 
 * @param   name        Descriptive block instance name. 
 * @param   nodeId      Unique identifier for block owner (typically a node id). 
 * @param   blockId     Unique identifier for block instance. 
 * @param   pipeEnviron Optional object to provide external services to the block. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Construct(kPxBlock* block, kType type, const kChar* name, k32u nodeId, k32u blockId, kPxEnviron pipeEnviron, kAlloc allocator); 

/**
 * Assigns this block to a processing group. 
 * 
 * By default, blocks in concurrent pipe environments are independently scheduled for execution. The kPxBlock_AssignGroup 
 * function can be used to create larger processing groups, so that multiple blocks are executed together. This technique can 
 * be used to optimize pipe execution by reducing context switching overhead, particulary when dealing with lightweight 
 * blocks. 
 * 
 * This function is typically called from within the pipe framework. However, this function call also be used 
 * when unit-testing blocks.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.
 * @param   groupId     Processing group id. 
 * @return              Operation status.
 */
kFsFx(kStatus) kPxBlock_AssignGroup(kPxBlock block, k32u groupId);

/**
* Reports whether this block is assigned to a processing group. 
* 
* This method is thread-safe while the pipe is running.
*
* @public              @memberof kPxBlock
* @param   block       Block object.
* @return              kTRUE if assigned to a group; kFALSE otherwise. 
*/
kFsFx(kBool) kPxBlock_IsGrouped(kPxBlock block);

/**
 * Gets the processing group id associated with this block. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.
 * @return              Processing group id (undefined if ungrouped). 
 */
kFsFx(k32u) kPxBlock_GroupId(kPxBlock block);

/** 
 * Sets a callback to receive messages sent by this block. 
 * 
 * This function is typically used within the pipe execution environment to receive messages 
 * from a block.  However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object. 
 * @param   function    Callback function.
 * @param   receiver    Callback receiver. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_SetSendHandler(kPxBlock block, kCallbackFx function, kPointer receiver); 

/** 
 * Sets a callback to provide notifications when this block drops a message. 
 * 
 * This function is typically used within the pipe execution environment to receive drop 
 * notifications from a block.  However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object. 
 * @param   function    Callback function.
 * @param   receiver    Callback receiver. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_SetDropHandler(kPxBlock block, kCallbackFx function, kPointer receiver); 

/** 
 * Defines an input port for a block. 
 *
 * This function should typically be called from within a block's OnSetup method. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @param   name        Port name.   
 * @param   id          Port identifier, unique within this block.    
 * @param   port        Returns the port object (optional).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_DefineInput(kPxBlock block, const kChar* name, k32u id, kPxPort* port);

/** 
 * Defines an output port for a block. 
 *
 * This function should typically be called from within a block's OnSetup method.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @param   name        Port name.   
 * @param   id          Port identifier, unique within this block.    
 * @param   port        Returns the port object (optional).   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_DefineOutput(kPxBlock block, const kChar* name, k32u id, kPxPort* port);

/**
* Enables the use of queued input ports. 
*
* By default, input messages are passed directly to the kPxBlock_OnReceive handler. Alternatively, 
* the kPxBlock_EnableQueuedInputs function can be used to enable message queues on each input port.  
* In this case, the 'msg' argument passed to the kPxBlock_OnReceive will be null. Queued inputs can 
* be traversed using the kPxPort_MessageCount, kPxPort_MessageAt, and kPxPort_MessageSourceAt functions. 
* Queued inputs can be accepted using the kPxPort_RemoveMessage function. 
* 
* When queued inputs are enabled, it is the block's responsibility to <em>eventually</em> process queued 
* inputs -- there are no automatic limits to constrain the amount of enqueued data.  However, any unprocessed 
* messages will be destroyed if the block is paused or stopped. 
* 
* This method can be called only from within a block's OnSetup method. 
*
* @public              @memberof kPxBlock
* @param   block       Block object.
* @param   enabled     kTRUE to enabled queued ports; kFALSE otherwise. 
* @return              Operation status.
*/
kFsFx(kStatus) kPxBlock_EnableQueuedInputs(kPxBlock block, kBool enabled);

/** 
 * Returns the block's internal settings. 
 *
 * The settings object returned by this function should only be modified by the pipe engine or 
 * within the block's OnSetup method.  
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Internal block settings. 
 */
kFsFx(kXml) kPxBlock_Settings(kPxBlock block);

/** 
 * Executes a block's OnSetup method. 
 *
 * This function is typically used within the pipe execution environment to perform 
 * block setup.  However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Setup(kPxBlock block);

/** 
 * Executes a block's OnStart method. 
 *
 * This function is typically used within the pipe execution environment to perform 
 * block start actions.  However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Start(kPxBlock block);

/** 
 * Executes a block's OnEngage method. 
 *
 * This function is called by the pipe execution environment after Start, to signal that 
 * blocks may begin generating arbitrary, asynchronous output messages. Refer to kPxBlock_OnEngage for 
 * information on overriding the OnEngage virtual method.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Engage(kPxBlock block);

/** 
 * Executes a block's OnStop method. 
 *
 * This function is typically used within the pipe execution environment to perform 
 * block stop actions. However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Stop(kPxBlock block);

/** 
 * Executes a block's OnReceive method. 
 *
 * This function is typically used within the pipe execution environment to perform 
 * block processing actions. However, this function call also be used when unit-testing blocks. 
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @param   port        Port for which message is intended. 
 * @param   msgInfo     Message object (readonly; ownership transferred). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Process(kPxBlock block, kPxPort port, kMsgInfo msgInfo);

/** 
 * Virtual method that can be overridden to perform custom block setup.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_OnSetup(kPxBlock block);

/** 
 * Virtual method that can be overridden to perform custom actions when a block is started. 
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_OnStart(kPxBlock block);

/** 
 * Virtual method that can be overridden to perform custom actions when a block is engaged. 
 *
 * Blocks are engaged shortly after being started. Once engaged, blocks are permitted to emit arbitrary 
 * asynchronous output messages (assuming that asynchronous messaging has been enabled on the block's 
 * output ports).
 * 
 * Blocks that only emit synchronous output messages, or that only emit asynchronous messages in response 
 * to received input messages, will not typically need to override OnEngage. However, if a block can emit 
 * spontaneous asynchronous messages (e.g., messages generated by a periodic timer thread, irrespective of 
 * any received messages), doing so before OnEngage will result in undefined behavior. As such, OnEngage
 * may provide a convenient location to start asynchronous threads owned by the block. (Side note: blocks that 
 * create threads must respect the pipe's thread priority/affinity configuration, which can be determined 
 * by calling kPxBlock_ThreadPriorityClass, kPxBlock_ThreadPriorityOffset, and kPxBlock_ThreadAffinity). 
 * 
 * The implementation of OnEngage is required to be thread-safe. Specifically, OnEngage may be invoked by the 
 * pipe framework simultaneous to the invocation of OnReceive. As such, the implementation of the OnEngage 
 * method should usually be limited to calling other thread-safe methods. 
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 * @see                 kPxBlock_Send, kPxPort_EnableAsynchronousData
 */
kFsFx(kStatus) kPxBlock_OnEngage(kPxBlock block);

/** 
 * Virtual method that can be overridden to perform custom actions when a block is stopped. 
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_OnStop(kPxBlock block);

/** 
 * Virtual method that can be overridden to perform custom actions when a messages is received. 
 *
 * This function is not invoked directly by external code; the kPxBlock_Process method should be called 
 * to provide a message to a block for processing.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @param   port        Port on which message was received. 
 * @param   msg         Message object (readonly; ownership is transferred; will be null if input queues are enabled). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_OnReceive(kPxBlock block, kPxPort port, kMsgSet msg);

/** 
 * Gets the full path for a given file name used by this block. 
 *
 * This function should be used to form fully-qualified file names when accessing files from 
 * within a block. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @param   fileName    File name. 
 * @param   filePath    Receives fully-qualified file path. 
 * @param   capacity    Capacity of the filePath argument. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_FilePath(kPxBlock block, const kChar* fileName, kChar* filePath, kSize capacity); 

/** 
 * Gets a reference to a named object. 
 *
 * The object returned by this function should be treated as read-only and should be disposed when 
 * no longer needed. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @param   name        Object name. 
 * @param   object      Receives object reference.
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_FindVariable(kPxBlock block, const kChar* name, kObject* object); 

/** 
 * Sends a message on a specific port.
 *
 * By default, the kPxPort EmitAsynchronousData property is set to kFALSE, meaning that messages can only be 
 * emitted synchronously. In this state, attempting to send a message outside the scope of a block's OnReceive 
 * handler will result in undefined behaviour. When the EmitAsynchronousData property is set to kTRUE, messages 
 * are delivered to the pipe asynchronously, enabling messages to be sent either inside or outside of a block's 
 * OnReceive handler. 
 * 
 * Asynchronous messages are not processed during the current synchronous pipe execution sequence; instead, 
 * they are queued and processed in a future execution pass. (For the serial pipe, a synchronous execution 
 * sequence refers to the processing of messages exchanged between connected blocks within the same pipe; for the parallel 
 * pipe, it refers to the processing of messages exchanged between connected blocks within the same processing group.)
 * As a result, the use of asynchronous messaging can alter the typical execution order of blocks within the pipe.
 * 
 * Asynchronous messages should not be sent before the pipe is ready to receive those messages. For blocks that 
 * generate asynchronous messages in response to input messages that they receive (e.g., blocks that combine 
 * multiple inputs to later produce a single output), the Send method can be safely called at any time after the first 
 * synchronous message has been received; no special consideration is required. However, for blocks that generate 
 * spontaneous asynchronous messages (e.g., blocks that generate output messages based on an internal timer), 
 * care should be taken to ensure that output messages are not sent before the block's Engage 
 * method has been invoked by the pipe environment. This can normally be accomplished by overriding the block's 
 * VOnEngage method, and deferring the start of any asynchronous threads to that method. (Side note: blocks that create 
 * threads must respect the pipe's thread priority/affinity configuration, which can be determined by calling 
 * kPxBlock_ThreadPriorityClass, kPxBlock_ThreadPriorityOffset, and kPxBlock_ThreadAffinity). 
 * 
 * Asynchronous messages should never be sent after a block's VOnStop method has been called. 
 * 
 * @public              @memberof kPxBlock
 * @param   block       Block object.   
 * @param   port        Port object.   
 * @param   msg         Output message object (ownership is transferred, irrespective of operation status).   
 * @return              Operation status. 
 * @see                 kPxPort_EnableAsynchronousData, kPxBlock_OnEngage
 */
kFsFx(kStatus) kPxBlock_Send(kPxBlock block, kPxPort port, kMsgSet msg);

/** 
 * Prints a diagnostic message. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @param   format      Format string.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_Printf(kPxBlock block, const kChar* format, ...);

/** 
 * Notifies the processing environment that this block has dropped data. 
 *
 * This function can be called by a block implementation to inform the processing environment
 * that data has been dropped.  In response, the processing environment will increment appropriate 
 * diagnostic counters. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kPxBlock_NotifyDrop(kPxBlock block); 

/** 
 * Gets an allocator that should be used to construct output messages. 
 *
 * Use of the message allocator is technically optional. However, use of this allocator can 
 * provide faster allocation and improved memory management. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Message allocator. 
 */
kFsFx(kAlloc) kPxBlock_MessageAlloc(kPxBlock block); 

/** 
 * Gets a health service provider that can be used when constructing health or profiling probes.
 *
 * The object returned by this function can be used as the 'provider' argument to the kHealthProbe_ConstructEx
 * or kProfileProbe_ConstructEx functions. This enables probes to be registered to a health service that is 
 * specific to this pipe environment. If a provider is not specified when constructing probes, the probes will 
 * instead be associated with the default system health provider.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Health service (may be null). 
 */
kFsFx(kHealth) kPxBlock_Health(kPxBlock block); 

/** 
 * Within an embedded node environment, this method can be used to access the current
 * FireSync synchnronization time. 
 *
 * Use of this method is not recommended, as it can produce unexpected results during 
 * simulation/testing.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Current FireSync time (us). 
 */
kFsFx(k64u) kPxBlock_Timestamp(kPxBlock block); 

/** 
 * Returns information about the source of the message currently being processed. 
 *
 * This function can only be called from within a block's OnReceive method.
 * 
 * The result of this function will be undefined if input messages queues are enabled.
 * 
 * @protected           @memberof kPxBlock
 * @param   block       Block object.   
 * @return              Message source identifier. 
 */
kFsFx(kMsgSource) kPxBlock_Sender(kPxBlock block); 

/** 
 * Gets a descriptive name for the block instance. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.  
 * @return              Descriptive instance name. 
 */
kFsFx(const kChar*) kPxBlock_Name(kPxBlock block);

/** 
 * Gets the unique numeric identifier for this block instance. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.  
 * @return              Block id. 
 */
kFsFx(k32u) kPxBlock_Id(kPxBlock block);

/** 
 * Finds the port object corresponding to the given id. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.  
 * @param   id          Port id.
 * @param   port        Receives port obect. 
 * @return              Operation status (kERROR_NOT_FOUND if not found). 
 */
kFsFx(kStatus) kPxBlock_FindPort(kPxBlock block, k32u id, kPxPort* port); 

/** 
 * Gets the number of ports belonging to the block. 
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.  
 * @return              Returns the number of ports belonging to the block. 
 */
kFsFx(kSize) kPxBlock_PortCount(kPxBlock block);

/** 
 * Gets the port at the specified index.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.  
 * @param   index       Port index. 
 * @return              Returns the port at the specified index.
 */
kFsFx(kPxPort) kPxBlock_PortAt(kPxBlock block, kSize index);

/**
* For blocks with queued inputs, reports whether each input has at least one message available. 
*
* This function can only be called from within a block's OnReceive method.
* 
* @public              @memberof kPxBlock
* @param   block       Block object.
* @return              kTRUE if at least one message is available; kFALSE otherwise.
*/
kFsFx(kBool) kPxBlock_HasAllInputs(kPxBlock block);

/**
 * Reports the configured pipe thread priority class. 
 * 
 * If a block creates any additional threads, those threads should be configured
 * to use the priority class reported by this method.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public          @memberof kPxBlock
 * @param   block   Block object.
 * @return          Thread priority class. 
 * @see             kThread_SetPriority
 */
kFsFx(kThreadPriorityClass) kPxBlock_ThreadPriorityClass(kPxBlock block);

/**
 * Reports the configured pipe thread priority offset. 
 *
 * If a block creates any additional threads, those threads should be configured
 * to use the priority offset reported by this method.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public          @memberof kPxBlock
 * @param   block   Block object.
 * @return          Thread priority offset. 
 * @see             kThread_SetPriority
 */
kFsFx(k32s) kPxBlock_ThreadPriorityOffset(kPxBlock block);

/**
 * Reports the configured pipe thread affinity. 
 *
 * If a block creates any additional threads, those threads should be configured
 * to use the thread affinity reported by this method.
 *
 * This method is thread-safe while the pipe is running.
 *
 * @public              @memberof kPxBlock
 * @param   block       Block object.
 * @return              Thread affinity (may be kNULL, signifying no affinity; do not modify the returned object).
 * @see                 kThread_SetAffinity
 */
kFsFx(kBitArray) kPxBlock_ThreadAffinity(kPxBlock block);

#include <kFireSync/Pipe/kPxBlock.x.h>

#endif
