/** 
* @file    kHealthLog.x.h
*
* @internal
* Copyright (C) 2019-2022 by LMI Technologies Inc. All rights reserved.
*/
#ifndef K_FIRESYNC_HEALTH_LOG_X_H
#define K_FIRESYNC_HEALTH_LOG_X_H

#include <kApi/Data/kList.h>
#include <kApi/Data/kMap.h>
#include <kApi/Io/kPath.h>

/**
* @struct  kHealthLogStatInfo
* @extends kValue
* @ingroup kFireSync-Health
* @brief   Metadata describing a health indicator stat.
*/
typedef struct kHealthLogStatInfo
{
    kText64 name;           ///< Descriptive label (e.g. "Event/0/QueueCount").
    kHealthId id;           ///< Health stat identifier (or kHEALTH_ID_NULL). 
    k32u instance;          ///< Distinguishes multiple health stats with same id.
    kBool shouldRestore;    ///< Should the previous value be reloaded at log construction time?
} kHealthLogStatInfo;

kDeclareValueEx(kFs, kHealthLogStatInfo, kValue)

/**
 * @class   kHealthLogManifest
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Represents a log manifest file, which lists health stat metadata.
 */

#define kHEALTH_LOG_MANIFEST_FIRST_VERSION      (1)         //first supported manifest version
#define kHEALTH_LOG_MANIFEST_LATEST_VERSION     (1)         //latest known manifest version

typedef struct kHealthLogManifestClass
{
    kObjectClass base;

    kArrayList statInfo;        //stat metadata values, sorted by GUID -- kArrayList<kHealthLogStatInfo>

} kHealthLogManifestClass;

kDeclareClassEx(kFs, kHealthLogManifest, kObject)

kFsFx(kStatus) kHealthLogManifest_Construct(kHealthLogManifest* manifest, kArrayList statInfo, kAlloc allocator);

kFsFx(kStatus) xkHealthLogManifest_Init(kHealthLogManifest manifest, kType type, kArrayList statInfo, kAlloc alloc);
kFsFx(kStatus) xkHealthLogManifest_VRelease(kHealthLogManifest manifest);

kFsFx(kStatus) kHealthLogManifest_Load(kHealthLogManifest* manifest, const kChar* path, kAlloc allocator);
kFsFx(kStatus) xkHealthLogManifest_Read(kHealthLogManifest manifest, kSerializer reader); 

kFsFx(kStatus) kHealthLogManifest_Save(kHealthLogManifest manifest, const kChar* path);
kFsFx(kStatus) xkHealthLogManifest_Write(kHealthLogManifest manifest, kSerializer writer);

kInlineFx(kArrayList) kHealthLogManifest_StatInfo(kHealthLogManifest manifest)
{
    return xkHealthLogManifest_Cast(manifest)->statInfo; 
}

/**
 * @class   kHealthLogSnapshot
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Represents a log snapshot file, which lists health stat values.
 */

#define kHEALTH_LOG_SNAPSHOT_FIRST_VERSION              (0)         //first supported snapshot version
#define kHEALTH_LOG_SNAPSHOT_LATEST_VERSION             (0)         //latest known snapshot version

#define kHEALTH_LOG_SNAPSHOT_ENCODING_GROUP_LENGTH      (4)         //number of stats packed into one encoding group
#define kHEALTH_LOG_SNAPSHOT_ENCODING_FIELD_WIDTH       (2)         //control bits per stat in group encoding word

#define kHEALTH_LOG_SNAPSHOT_ENCODING_8_BIT             (0)         //8-bit encoding
#define kHEALTH_LOG_SNAPSHOT_ENCODING_16_BIT            (1)         //16-bit encoding
#define kHEALTH_LOG_SNAPSHOT_ENCODING_32_BIT            (2)         //32-bit encoding
#define kHEALTH_LOG_SNAPSHOT_ENCODING_64_BIT            (3)         //64-bit encoding

typedef struct kHealthLogSnapshotClass
{
    kObjectClass base;

    kHealthLog log;         //owner
    kArrayList stats;       //stat values in snapshot, sorted by GUID -- kArrayList<k64s>

} kHealthLogSnapshotClass;

kDeclareClassEx(kFs, kHealthLogSnapshot, kObject)

kFsFx(kStatus) kHealthLogSnapshot_Construct(kHealthLogSnapshot* snapshot, kHealthLog log, kArrayList stats, kAlloc allocator);

kFsFx(kStatus) xkHealthLogSnapshot_Init(kHealthLogSnapshot snapshot, kType type, kHealthLog log, kArrayList stats, kAlloc alloc);
kFsFx(kStatus) xkHealthLogSnapshot_VRelease(kHealthLogSnapshot snapshot);

kFsFx(kStatus) kHealthLogSnapshot_Load(kHealthLogSnapshot* snapshot, kHealthLog log, k32u id, kAlloc allocator);
kFsFx(kStatus) xkHealthLogSnapshot_Read(kHealthLogSnapshot snapshot, kSerializer reader);
kFsFx(kStatus) xkHealthLogSnapshot_ReadStats(kHealthLogSnapshot snapshot, kSerializer reader, kSize statCount);

kFsFx(kStatus) kHealthLogSnapshot_Save(kHealthLogSnapshot manisnapshotfest, k32u id);
kFsFx(kStatus) xkHealthLogSnapshot_Write(kHealthLogSnapshot snapshot, kSerializer writer); 
kFsFx(kStatus) xkHealthLogSnapshot_WriteStats(kHealthLogSnapshot snapshot, kSerializer reader);

kInlineFx(kArrayList) kHealthLogSnapshot_Stats(kHealthLogSnapshot snapshot)
{
    return xkHealthLogSnapshot_Cast(snapshot)->stats; 
}

kInlineFx(k32u) xkHealthLogSnapshot_Encoding(k64s value)
{
    if      ((k8S_MIN  <= value) && (value <= k8S_MAX))     return kHEALTH_LOG_SNAPSHOT_ENCODING_8_BIT;
    else if ((k16S_MIN <= value) && (value <= k16S_MAX))    return kHEALTH_LOG_SNAPSHOT_ENCODING_16_BIT; 
    else if ((k32S_MIN <= value) && (value <= k32S_MAX))    return kHEALTH_LOG_SNAPSHOT_ENCODING_32_BIT; 
    else                                                    return kHEALTH_LOG_SNAPSHOT_ENCODING_64_BIT;
}

/**
* @struct  kHealthLogSnapshotBlockEntry
* @extends kValue
* @ingroup kFireSync-Health
* @brief   Represents one snapshot entry in snapshot block header. 
*/
typedef struct kHealthLogSnapshotBlockEntry
{
    k32u snapshotId;            //snapshot identifier
    k16u offset;                //offset from beginning of block file to snapshot data (bytes)
    k16u size;                  //size of snapshot data (bytes)
} kHealthLogSnapshotBlockEntry;

kDeclareValueEx(kFs, kHealthLogSnapshotBlockEntry, kValue)

/**
 * @class   kHealthLogSnapshotBlock
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Represents a group of snapshots that are managed as a single file. 
 */

#define kHEALTH_LOG_SNAPSHOT_BLOCK_VERSION          (0)

typedef struct kHealthLogSnapshotBlockClass
{
    kObjectClass base;

    kHealthLogSnapshotManager manager;      //owner

    k32u id;                                //block identifier
    k32u revision;                          //current block revision (k32U_NULL if no review written yet)

    kArrayList entries;                     //list of snapshot entries -- kArrayList<kHealthLogSnapshotBlockEntry>
    k32u totalEntrySize;                    //total size of all snapshot entries

} kHealthLogSnapshotBlockClass;

kDeclareClassEx(kFs, kHealthLogSnapshotBlock, kObject)

kFsFx(kStatus) kHealthLogSnapshotBlock_Construct(kHealthLogSnapshotBlock* block, kHealthLogSnapshotManager manager, k32u id, kAlloc allocator);

kFsFx(kStatus) xkHealthLogSnapshotBlock_Init(kHealthLogSnapshotBlock block, kType type,  kHealthLogSnapshotManager manager, k32u id, kAlloc alloc);
kFsFx(kStatus) xkHealthLogSnapshotBlock_VRelease(kHealthLogSnapshotBlock block);

kFsFx(kStatus) kHealthLogSnapshotBlock_Load(kHealthLogSnapshotBlock* block, kHealthLogSnapshotManager manager, k32u id, k32u revision, kAlloc allocator);

kFsFx(kSize) kHealthLogSnapshotBlock_SnapshotCount(kHealthLogSnapshotBlock block);
kFsFx(k32u) kHealthLogSnapshotBlock_SnapshotAt(kHealthLogSnapshotBlock block, kSize index);

kFsFx(kStatus) kHealthLogSnapshotBlock_DeleteSnapshot(kHealthLogSnapshotBlock block, k32u snapshotId);
kFsFx(kStatus) kHealthLogSnapshotBlock_FlushDeletion(kHealthLogSnapshotBlock block);
kFsFx(kStatus) kHealthLogSnapshotBlock_RewriteBlock(kHealthLogSnapshotBlock block, k32u snapshotId, const void* data, kSize size);

kFsFx(kStatus) kHealthLogSnapshotBlock_AddSnapshot(kHealthLogSnapshotBlock block, k32u snapshotId, const void* data, kSize size);

kFsFx(kStatus) kHealthLogSnapshotBlock_LoadSnapshot(kHealthLogSnapshotBlock block, k32u snapshotId, void* data, kSize* size, kAlloc alloc);

kFsFx(k32u) kHealthLogSnapshotBlock_Size(kHealthLogSnapshotBlock block);
kFsFx(k32u) kHealthLogSnapshotBlock_EstimateSize(kHealthLogSnapshotBlock block, k32u entrySize);
kFsFx(kStatus) kHealthLogSnapshotBlock_SnapshotSize(kHealthLogSnapshotBlock block, k32u id, k32u* size);

kFsFx(k32u) xkHealthLogSnapshotBlock_HeaderSize(kHealthLogSnapshotBlock block, kSize entryCount);
kFsFx(k32u) xkHealthLogSnapshotBlock_HeaderContentSize(kHealthLogSnapshotBlock block, kSize entryCount);

kFsFx(kStatus) xkHealthLogSnapshotBlock_ReadHeader(kHealthLogSnapshotBlock block, k32u revision);

kFsFx(kStatus) xkHealthLogSnapshotBlock_FindEntry(kHealthLogSnapshotBlock block, k32u snapshotId, kSize* index);

kFsFx(kStatus) xkHealthLogSnapshotBlock_WriteHeader(kHealthLogSnapshotBlock block, kArrayList newEntries, kMemory stream);

/**
 * @class   kHealthLogSnapshotManager
 * @extends kObject
 * @ingroup kFireSync-Health
 * @brief   Organizes the saving/loading of snapshot data using a set of snapshot block files. 
 */

typedef struct kHealthLogSnapshotManagerClass
{
    kObjectClass base;

    kChar path[kPATH_MAX];              //snapshot directory
    k32u maxBlockSize;                  //maximum snapshot block file size

    kArrayList blocks;                  //list of snapshot blocks -- kArrayList<kHealthLogSnapshotBlock>
    k32u nextBlockId;                   //next available block id

    kSize dirtyBlockIndex;              //index of block with an uncommitted deletion, if one exists; otherwise, kSIZE_NULL
    
    kMap idToBlockIndexMap;             //maps from snapshot id to block index -- kMap<k32u, kSize>

    k32u totalSize;                     //total current size of all blocks 

} kHealthLogSnapshotManagerClass;

kDeclareClassEx(kFs, kHealthLogSnapshotManager, kObject)

kFsFx(kStatus) kHealthLogSnapshotManager_Construct(kHealthLogSnapshotManager* manager, const kChar* path, k32u maxBlockSize, kAlloc allocator);

kFsFx(kStatus) xkHealthLogSnapshotManager_Init(kHealthLogSnapshotManager manager, kType type, const kChar* path, k32u maxBlockSize, kAlloc alloc);
kFsFx(kStatus) xkHealthLogSnapshotManager_VRelease(kHealthLogSnapshotManager manager);

kFsFx(kStatus) kHealthLogSnapshotManager_EnumerateSnapshots(kHealthLogSnapshotManager manager, kArrayList* snapshotIds);
kFsFx(kStatus) kHealthLogSnapshotManager_DeleteSnapshot(kHealthLogSnapshotManager manager, k32u id);

kFsFx(kStatus) kHealthLogSnapshotManager_SaveSnapshot(kHealthLogSnapshotManager manager, k32u id, const void* data, kSize size);

kFsFx(kStatus) kHealthLogSnapshotManager_LoadSnapshot(kHealthLogSnapshotManager manager, k32u id, void* data, kSize* size, kAlloc alloc);
kFsFx(kStatus) kHealthLogSnapshotManager_SnapshotSize(kHealthLogSnapshotManager manager, k32u id, k32u* size);

kFsFx(k32u) kHealthLogSnapshotManager_Size(kHealthLogSnapshotManager manager);

kFsFx(kStatus) xkHealthLogSnapshotManager_Load(kHealthLogSnapshotManager manager);
kFsFx(kStatus) xkHealthLogSnapshotManager_BuildRevisionMap(kHealthLogSnapshotManager manager, kMap revisionMap);
kFsFx(kStatus) xkHealthLogSnapshotManager_BlockPath(kHealthLogSnapshotManager manager, k32u id, k32u revision, kChar* path, kSize capacity);
kFsFx(kStatus) xkHealthLogSnapshotManager_LoadBlocks(kHealthLogSnapshotManager manager, kMap revisionMap);

kFsFx(kStatus) xkHealthLogSnapshotManager_FlushDirtyBlock(kHealthLogSnapshotManager manager);

kFsFx(kStatus) xkHealthLogSnapshotManager_AddToDirtyBlock(kHealthLogSnapshotManager manager, k32u id, const void* data, kSize size);
kFsFx(kStatus) xkHealthLogSnapshotManager_AddToAnyBlock(kHealthLogSnapshotManager manager, k32u id, const void* data, kSize size);
kFsFx(kStatus) xkHealthLogSnapshotManager_AddToNewBlock(kHealthLogSnapshotManager manager, k32u id, const void* data, kSize size);

/**
* @struct  kHealthLogLevel
* @extends kValue
* @ingroup kFireSync-Health
* @brief   Represents information about a pruning level.
*/
typedef struct kHealthLogLevel
{
    kListItem begin;                    //points to location within snaphotIds that represents start of a level (ascending/foward order)
    kSize count;                        //count of items at this pruning level
} kHealthLogLevel;

kDeclareValueEx(kFs, kHealthLogLevel, kValue)

/*
 * kHealthLog class
*/

#define xkHEALTH_LOG_CLEAR_FILE_NAME    "clear"      //special file name used to signal that the log should be cleared on reload

#define xkHEALTH_LOG_MIN_BLOCK_SIZE         (2048)      //minimum size of snapshot block; somewhat arbitrary, but reasonable based on expected snapshot sizes
#define xkHEALTH_LOG_DEFAULT_BLOCK_SIZE     (4096)      //default size of snapshot block
#define xkHEALTH_LOG_MAX_BLOCK_SIZE         (32768)     //maximum size of snapshot block; chosen to work with 16b offset/size fields in snapshot blocks

typedef struct kHealthLogClass
{
    kObjectClass base;

    kChar directory[kPATH_MAX];         //health log root directory

    k32u fileLimit;                     //constraint on maximum count of snapshot files in the log
    k32u sizeLimit;                     //constraint on maximum cumulative size of all snapshot files in the log, in bytes
    k32u blockSize;                     //maximum size per log file, in bytes

    //protected by logLock
    kHealthLogSnapshotManager snapshotManager;
    k32u nextManifestId;                //next id to use for a new manifest file
    k32u nextSnapshotId;                //next id to use for a new snapshot file
    kList snapshotIds;                  //sorted list (ascending) of snapshot numbers that currently exist in the log -- kList<k32u>
    kList pruningLevels;                //one entry per pruning level; empty if snapshotIds empty; otherwise, pruningLevels[0]->begin is snapshotIds->first -- kList<kHealthLogLevel>
    kLock logLock;                      //provides access to snapshot-related data
    
    //protected by statLock
    kArrayList statInfo;                //list of stat metadata, sorted by 16-bit GUID  -- kArrayList<kHealthLogStatInfo>
    kBool statInfoChanged;              //if true, statInfo has changed; requires new manifest to be written
    kArrayList statValues;              //list of stat values, sorted by 16-bit GUID -- kArrayList<k64u>
    kMap idToGuidMap;                   //provides lookup from 64-bit health id-instance pair to 16-bit log GUID -- kMap<k64u, k16u>
    kLock statLock;                     //provides quick access to stats             

} kHealthLogClass;

kDeclareClassEx(kFs, kHealthLog, kObject)

kFsFx(kStatus) xkHealthLog_Init(kHealthLog log, kType type, const kChar* directory, k32u fileLimit, k32u sizeLimit, k32u blockSize, kAlloc alloc);
kFsFx(kStatus) xkHealthLog_VRelease(kHealthLog log);

kFsFx(kStatus) xkHealthLog_ClearPath(kHealthLog log, kChar* path, kSize capacity);
kFsFx(kStatus) xkHealthLog_ManifestPath(kHealthLog log, k32u manifestId, kChar* path, kSize capacity);
kFsFx(kStatus) xkHealthLog_SnapshotDirectory(kHealthLog log, kChar* path, kSize capacity);

kInlineFx(k64u) xkHealthLog_Key(kHealthId id, k32u instance)
{
    return (((k64u)instance) << 32) | ((k64u) id);
}

kInlineFx(kStatus) xkHealthLog_FindGuid(kHealthLog log, kHealthId id, k32u instance, k16u* guid)
{
    kObj(kHealthLog, log); 
    k64u key = xkHealthLog_Key(id, instance);

    return kMap_FindT(obj->idToGuidMap, &key, guid);
}

kFsFx(kStatus) xkHealthLog_Load(kHealthLog log); 
kFsFx(kStatus) xkHealthLog_AttemptLoad(kHealthLog log);
kFsFx(kStatus) xkHealthLog_ProcessClearRequest(kHealthLog log);
kFsFx(kStatus) xkHealthLog_WipeLogFiles(kHealthLog log);

kFsFx(kStatus) xkHealthLog_EstablishDirectoryStructure(kHealthLog log); 
kFsFx(kStatus) xkHealthLog_EstablishSnapshotManager(kHealthLog log);

kFsFx(kStatus) xkHealthLog_ReviewManifests(kHealthLog log);
kFsFx(kStatus) xkHealthLog_EstablishValidManifests(kHealthLog log, kArrayList* list);
kFsFx(kStatus) xkHealthLog_VerifyManifestFile(kHealthLog log, const kChar* fileName, k32u* id);
kFsFx(kStatus) xkHealthLog_EstablishManifestBackup(kHealthLog log, k32u* primaryId, kArrayList backupCandidateIds);
kFsFx(kStatus) xkHealthLog_CompareFiles(kHealthLog log, const kChar* pathA, const kChar* pathB);
kFsFx(kStatus) xkHealthLog_LoadManifest(kHealthLog log, k32u manifestId);

kFsFx(kStatus) xkHealthLog_ReviewSnapshots(kHealthLog log);
kFsFx(kStatus) xkHealthLog_EstablishSnapshots(kHealthLog log, kArrayList* snapshotList);
kFsFx(kStatus) xkHealthLog_EstablishLatestSnapshot(kHealthLog log, kArrayList snapshotList);
kFsFx(kStatus) xkHealthLog_VerifySnapshotFile(kHealthLog log, k32u id);
kFsFx(kStatus) xkHealthLog_LoadSnapshot(kHealthLog log, k32u snapshotId);

kFsFx(kStatus) xkHealthLog_EstablishPruningLevels(kHealthLog log);
kFsFx(kStatus) xkHealthLog_FindPruningLevelStart(kList snapshots, kSize levelIndex, kListItem* iterator, kSize* index);

kFsFx(kStatus) xkHealthLog_ReconcileManifest(kHealthLog log);

kFsFx(kStatus) xkHealthLog_AddStat(kHealthLog log, kHealthId id, k32u instance, const kChar* name, kBool shouldRestore, k64s value);

kFsFx(kStatus) xkHealthLog_BuildSnapshot(kHealthLog log, kArrayList statList, kHealthLogSnapshot* snapshot, kHealthLogSnapshot* manifest);
kFsFx(kStatus) xkHealthLog_SaveManifest(kHealthLog log, kHealthLogManifest manifest, k32u manifestId);
kFsFx(kStatus) xkHealthLog_SaveSnapshot(kHealthLog log, kHealthLogSnapshot snapshot, k32u snapshotId);
kFsFx(kStatus) xkHealthLog_AddSnapshotId(kHealthLog log, k32u snapshotId); 

kFsFx(kStatus) xkHealthLog_Prune(kHealthLog log);
kFsFx(kBool) xkHealthLog_ShouldPrune(kHealthLog log);
kFsFx(kStatus) xkHealthLog_PruneNext(kHealthLog log);
kFsFx(kStatus) xkHealthLog_SelectPruningLevel(kHealthLog log, kListItem* levelIt);
kFsFx(kStatus) xkHealthLog_DeleteSnapshot(kHealthLog log, k32u snapshotId);

kFsFx(kStatus) xkHealthLog_InitStatHistories(kHealthLog log, kHealthSummary summary, kArrayList* statHistories); 
kFsFx(kStatus) xkHealthLog_AddSnapshotToSummary(kHealthLog log, k32u snapshotId, kArrayList statHistories);

kFsFx(kHealthLogSnapshotManager) xkHealthLog_SnapshotManager(kHealthLog log);

kFsFx(kStatus) xkHealthLog_VerifyCrc(kMemory stream, kSerializer reader);
kFsFx(kStatus) xkHealthLog_AppendCrc(kMemory stream, kSerializer writer);

//exposed for unit testing
kFsFx(kStatus) kHealthLog_SnaphotIds(kHealthLog log, kArrayList snapshotList);

int xkHealthLog_32uComparator(const void* a, const void* b);

#endif
