/** 
 * @file    kTimeAvg.h
 * @brief   Declares the kTimeAvg type. 
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_TIME_AVG_H
#define K_FIRESYNC_TIME_AVG_H

#include <kFireSync/kFsDef.h>
#include <kFireSync/Data/kTimeAvg.x.h>

/**
 * @class   kTimeAvg
 * @extends kObject
 * @ingroup kFireSync-Data
 * @brief   Helper class to support calculation of running averages over a time window.
 */
//typedef kObject kTimeAvg;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kTimeAvg object.
 *
 * @public              @memberof kTimeAvg
 * @param   stat        Destination for the constructed object handle. 
 * @param   window      Sample retention time window, in microseconds (minimum 1). 
 * @param   minInterval Minimum elasped time before a new sample should be accepted (microseconds; minimum 1).
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kTimeAvg_Construct(kTimeAvg* stat, k64u window, k64u minInterval, kAlloc allocator); 

/** 
 * Clears dataset.
 *
 * @public              @memberof kTimeAvg
 * @return              Operation status; kERROR_PARAMETER if sample is not accepted.  
 */
kFsFx(kStatus) kTimeAvg_Clear(kTimeAvg stat); 

/** 
 * Adds a sample value to the data set.
 *
 * @public              @memberof kTimeAvg
 * @param   stat        kTimeAvg object.  
 * @param   timestamp   Time corresponding to sample value (microseconds).
 * @param   value       Sample value to add.   
 * @return              Operation status; kERROR_PARAMETER if sample is not accepted.  
 */
kFsFx(kStatus) kTimeAvg_Add(kTimeAvg stat, k64u timestamp, k64s value); 

/** 
 * Reports the current count of samples the dataset
 *
 * @public              @memberof kTimeAvg
 * @param   stat        kTimeAvg object.  
 * @return              Sample count. 
 */
kFsFx(kSize) kTimeAvg_Count(kTimeAvg stat); 

/** 
 * Reports the sum of current sample values in the dataset.
 *
 * @public              @memberof kTimeAvg
 * @param   stat        kTimeAvg object.  
 * @return              Sum of sample values.
 */
kFsFx(k64s) kTimeAvg_ValueSum(kTimeAvg stat); 

/** 
 * Reports the time and value differences between the last and first samples.
 *
 * @public              @memberof kTimeAvg
 * @param   stat        kTimeAvg object.  
 * @param   timeDiff    Receives the time difference (microseconds) between the last and first samples in the dataset.
 * @param   valueDiff   Receives the value difference between the last and first samples in the dataset.  
 * @return              Operation status; kERROR_STATE if less than two samples.
 */
kFsFx(kStatus) kTimeAvg_SampleDiff(kTimeAvg stat, k64u* timeDiff, k64s* valueDiff); 

#endif
