/**
* @file    kSpline.h
* @brief   Declares the kSpline type.
*
* @internal
* Copyright (C) 2007-2022 by LMI Technologies Inc. All rights reserved.
*/

#ifndef K_FIRESYNC_SPLINE_H
#define K_FIRESYNC_SPLINE_H

#include <kFireSync/kFsDef.h>

/**
 * @class       kSpline
 * @extends     kObject
 * @ingroup     kFireSync-Data
 * @brief       Used to fit and apply splines.
 * 
 * At present, the implementation of this class supports only piecewise-linear splines. 
 */
//typedef kObject kSpline;          --forward-declared in kFsDef.x.h

/**
 * Constructs a kSpline object.
 *
 * @public              @memberof kSpline
 * @param   spline      Destination for the constructed object handle.
 * @param   knotCount   Optionally specifies the number of spline points for which to allocate space (can be zero). 
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status.
*/
kFsFx(kStatus) kSpline_Construct(kSpline* spline, kSize knotCount, kAlloc allocator);

/**
 * Calculates the output of the spline function for a single input value. 
 *
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @param   x           Input value.
 * @return              Spline output value.
*/
kFsFx(k64f) kSpline_Apply(kSpline spline, k64f x); 

/**
 * Calculates outputs of the spline function for an array of input values.
 *
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @param   x           Input values.
 * @param   f           Spline output values.
 * @param   count       Count of input/output values.
 * @return              Operation status.
*/
kFsFx(kStatus) kSpline_ApplyN(kSpline spline, const k64f* x, k64f* f, kSize count); 

/**
 * Sets the valid domain for the spline.
 *
 * If domain limits are set and an input value provided to the Apply (or ApplyN) function 
 * is outside of the domain, the output will be k64F_NULL.
 * 
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @param   xMin        Minimum domain value.
 * @param   xMax        Maxium domain value.
 * @return              Operation status.
*/
kFsFx(kStatus) kSpline_SetDomain(kSpline spline, k64f xMin, k64f xMax); 

/**
 * Gets the polynomial coefficients at the specified index. 
 *
 * During data fitting, one polynomial is generated for each adjacent pair of unique x-values. 
 * Accordingly, the number of generated polynomials will be one less than the knot count.  
 * 
 * The order of the polynomials is determined by the data-fitting function. N+1 coefficients 
 * will be generated per polynomial for an order N data fit. 
 *
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @param   index       Polynomial index. 
 * @return              Pointer to the requested polynomial coefficients (lowest-order coefficient first). 
*/
kFsFx(k64f*) kSpline_Coefficients(kSpline spline, kSize index); 

/**
 * Reports the count of spline knots.
 *
 * The number of spline knots is determined by the number of unique x-values provided 
 * during data fitting.
 *
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @return              Knot count.
*/
kFsFx(kSize) kSpline_KnotCount(kSpline spline); 

/**
* Gets the x-values corresponding to the spline knots.
*
* @public              @memberof kSpline
* @param   spline      Spline object.
* @return              Pointer to spline knots. 
*/
kFsFx(k64f*) kSpline_Knots(kSpline spline); 

/**
 * Gets the user-defined minimum x-value for the spline.
 *
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @return              Minimum x value.
*/
kFsFx(k64f) kSpline_DomainMin(kSpline spline); 

/**
* Gets the user-defined maximum x-value for the spline.
*
* @public              @memberof kSpline
* @param   spline      Spline object.
* @return              Maximum x value.
*/
kFsFx(k64f) kSpline_DomainMax(kSpline spline); 

/**
 * Generates spline coefficients for the given input data. 
 * 
 * The input data will be used to generate spline coefficients. The input data points 
 * will first be sorted by x-value. One spline knot will then be created for each 
 * unique x-value.  At least two unique values are required.  
 * 
 * At present, this class supports only first-order spline segments. 
 * 
 * @public              @memberof kSpline
 * @param   spline      Spline object.
 * @param   order       Polynomial order for individual spline segments (must be 1). 
 * @param   x           x (input) values.
 * @param   f           f (output) values.
 * @param   count       Count of fit points. 
 * @return              Operation status.
*/
kFsFx(kStatus) kSpline_Fit(kSpline spline, k32u order, const k64f* x, const k64f* f, kSize count);

#include <kFireSync/Data/kSpline.x.h>

#endif 

