/** 
 * @file    kAlgMsg.x.h
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ALG_MSG_X_H
#define K_FIRESYNC_ALG_MSG_X_H

/**
 * @internal
 * @struct  kAlgMsgDescriptor
 * @ingroup kFireSync-Data
 * @brief   Metadata that describes one part of a multipart algorithm message.
 */
typedef struct kAlgMsgDescriptor
{
    kType itemType;             ///< Item type of data part.
    kSize dimensionCount;       ///< Number of dimensions of data part. 
    kSize length[2];            ///< Dimension lengths of data part.
    kSize count;                ///< Total item count across all dimensions. 
    void* allocatedData;        ///< Pointer to data block allocated for part (will be kNULL if attached). 
} kAlgMsgDescriptor; 

typedef struct kAlgMsgClass
{
    kMsgSetClass base; 

    kSize frameCount;           //Count of frames in messsage.
    kSize partCount;            //Count of data parts per frame.

    kAlgMsgDescriptor descriptors[kALG_MSG_MAX_PARTS];         //Data part descriptors; one per part. 
    
    kPointer* parts;            //2D array of pointers to individual data parts [frameCount][partCount].
} kAlgMsgClass; 

kDeclareClassEx(kFs, kAlgMsg, kMsgSet)

/*
* Forward declarations
*/

kInlineFx(kType) kAlgMsg_PartType(kAlgMsg msg, kSize partIndex);
kInlineFx(void*) kAlgMsg_Part(kAlgMsg msg, kSize frameIndex, kSize partIndex);

/*
* Protected methods
*/
        
kFsFx(kStatus) kAlgMsg_Init(kAlgMsg msg, kType type, kAlloc alloc);
kFsFx(kStatus) kAlgMsg_VRelease(kAlgMsg msg);
kFsFx(kStatus) kAlgMsg_VClone(kAlgMsg msg, kAlgMsg other, kAlloc valueAlloc, kObject context); 
kFsFx(kSize) kAlgMsg_VSize(kAlgMsg msg);

/*
 * Child classes can optionally use these methods to simplify serialization implementation.
 */
kFsFx(kStatus) kAlgMsg_WriteDat6V0(kAlgMsg msg, kSerializer serializer);
kFsFx(kStatus) kAlgMsg_ReadDat6V0(kAlgMsg msg, kSerializer serializer);

kFsFx(kSize) kAlgMsg_VCount(kAlgMsg msg);
kFsFx(const kStamp*) kAlgMsg_VStampAt(kAlgMsg msg, kSize index);
kFsFx(kStatus) kAlgMsg_VSetStampAt(kAlgMsg msg, kSize index, const kStamp* stamp);
kFsFx(kStatus) kAlgMsg_VDataAt(kAlgMsg msg, kSize index, kArrayList* list);

/*
 * Child classes should use these methods to define the message content layout.
 */
kFsFx(kStatus) kAlgMsg_AddValuePart(kAlgMsg msg, kType type); 
kFsFx(kStatus) kAlgMsg_AddArray1Part(kAlgMsg msg, kType type, kSize length); 
kFsFx(kStatus) kAlgMsg_AddArray2Part(kAlgMsg msg, kType type, kSize length0, kSize length1); 
kFsFx(kStatus) kAlgMsg_SetFrameCount(kAlgMsg msg, kSize count); 

/*
 * Child classes should use these methods to attach external data pointers.
 */
kFsFx(kStatus) kAlgMsg_BeginAttach(kAlgMsg msg);
kFsFx(kStatus) kAlgMsg_AttachPart(kAlgMsg msg, kSize frameIndex, kSize partIndex, void* part);

/*
 * Child classes should use this method to allocate memory when *not* using attached data.
 */
kFsFx(kStatus) kAlgMsg_Allocate(kAlgMsg msg);

/*
 * Dellocates memory that was allocated for data content and part pointerss.
 */
kFsFx(kStatus) kAlgMsg_Clear(kAlgMsg msg); 

kInlineFx(kAlgMsgDescriptor*) xkAlgMsg_DescriptorAt(kAlgMsg msg, kSize partIndex)
{
    kObj(kAlgMsg, msg); 

    return &obj->descriptors[partIndex];
}

kInlineFx(kSize) xkAlgMsg_ItemSize(kAlgMsg msg, kSize partIndex)
{
    return kType_Size(xkAlgMsg_DescriptorAt(msg, partIndex)->itemType); 
}

kInlineFx(kSize) xkAlgMsg_ItemCount(kAlgMsg msg, kSize partIndex)
{
    return xkAlgMsg_DescriptorAt(msg, partIndex)->count; 
}

kInlineFx(kSize) xkAlgMsg_PartSize(kAlgMsg msg, kSize partIndex)
{
    return xkAlgMsg_ItemSize(msg, partIndex) * xkAlgMsg_ItemCount(msg, partIndex); 
}

kInlineFx(void*) xkAlgMsg_PartT(kAlgMsg msg, kSize frameIndex, kSize partIndex, kSize itemSize)
{
    kAssert(xkType_IsPointerCompatible(kAlgMsg_PartType(msg, partIndex), itemSize));  

    return kAlgMsg_Part(msg, frameIndex, partIndex);
} 

#endif
