/** 
 * @file    kCudaStream.h
 * @brief   Declares the kCudaStream class. 
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CUDA_STREAM_H
#define K_FIRESYNC_CUDA_STREAM_H

#include <kFireSync/Cuda/kCudaDef.h>
#include <kFireSync/Cuda/kCudaStream.x.h>

/**
 * @class   kCudaStream
 * @extends kObject
 * @ingroup kFireSync-Cuda
 * @brief   Represents a Cuda stream object. 
 */
//typedef kObject kCudaStream;               --forward-declared in kFsDef.x.h

/**
 * Constructs a kCudaStream object.
 *
 * @public              @memberof kCudaStream
 * @param   stream      Receives the constructed object.
 * @param   allocator   Memory allocator (or kNULL for default).
 * @return              Operation status.
 */
kFsFx(kStatus) kCudaStream_Construct(kCudaStream* stream, kObject allocator);

/** 
 * Sets the default synchronization mode of the stream. 
 * 
 * The default stream synchronization mode determines how operations are performed when 
 * a synchronization mode is not otherwise specified as an explicit argument to an operation (e.g., kObject_Clone). 
 * This setting only applies to kApi/kFireSync utility methods; the underlying Cuda runtime library does not 
 * support a default synchronization mode setting.
 * 
 * If this method is not called, the stream's default synchronization mode will be kCUDA_SYNC_WAIT_HOST.
 *
 * @public              @memberof kCudaStream
 * @param   stream      Cuda stream. 
 * @param   sync        Default synchronization mode. 
 * @return              Operation status.
 */
kInlineFx(kStatus) kCudaStream_SetDefaultSync(kCudaStream stream, kCudaSync sync)
{
    kObj(kCudaStream, stream); 

    //default stream behaviour cannot be modified
    if (stream != xkCudaStream_Default())
    {
        obj->sync = sync; 
    }

    return kOK;
}

/** 
 * Reports the default synchronization mode of the stream. 
 * 
 * @public              @memberof kCudaStream
 * @param   stream      Cuda stream. 
 * @return              Default synchronization mode. 
 */
kInlineFx(kCudaSync) kCudaStream_DefaultSync(kCudaStream stream)
{
    kObj(kCudaStream, stream); 

    return obj->sync;
}

/** 
 * Gets the Cuda stream handle (cudaStream_t).
 *
 * The return value of this function can be passed to Cuda kernels (4th parameter of kernel invocation).
 *
 * @public      @memberof kCudaStream
 * @return      Cuda runtime stream handle. 
 */
kInlineFx(kPointer) kCudaStream_Handle(kCudaStream stream)
{
    kObj(kCudaStream, stream);

    return obj->handle;
}

/**
 * Conditionally synchronizes a Cuda stream.
 * 
 * If kCUDA_SYNC_DEFAULT is provided as the 'sync' argument, then the synchronization mode will be resolved 
 * by examining the stream's default synchronization mode (kCudaStream_DefaultSync). 
 * 
 * Synchronization will be performed if the resolved synchronization mode is kCUDA_SYNC_WAIT, or if the 
 * resolved synchronization mode is kCUDA_SYNC_WAIT_HOST and the isHostOperation parameter is kTRUE.
 * 
 * If synchronization is performed, this function will block until all operations on the stream have been 
 * completed.
 *
 * @public                      @memberof kCudaStream
 * @param   stream              Cuda stream.
 * @param   sync                Synchronization mode (optional, defaults to kCUDA_SYNC_WAIT_HOST).
 * @param   isHostOperation     Does the most recent operation affect memory access from the host (optional, defaults to kTRUE)?
 * @return                      Operation status.
 */
kFsFx(kStatus) kCudaStream_Synchronize(kCudaStream stream, kCudaSync sync = kCUDA_SYNC_WAIT_HOST, kBool isHostOperation = kTRUE);

/**
 * Attaches serial managed memory to a stream.
 * 
 * Refer to @ref kCudaManagedAlloc for details on serial managed memory attachment. 
 * 
 * This method has no effect on other types of memory.
 *
 * @public          @memberof kCudaStream
 * @param   stream  Cuda stream. 
 * @param   mem     Memory to attach.  
 * @param   sync    Synchronization mode. 
 * @return          Operation status.
 * @see             kCudaManagedAlloc, kCudaStream_DetachMemory
 */
kFsFx(kStatus) kCudaStream_AttachMemory(kCudaStream stream, kPointer mem, kCudaSync sync = kCUDA_SYNC_DEFAULT);

/**
 * Detaches serial managed memory from a stream.
 *  
 * Refer to @ref kCudaManagedAlloc for details on serial managed memory attachment. 
 * 
 * This method has no effect on other types of memory.
 *
 * @public          @memberof kCudaStream
 * @param   stream  Cuda stream. 
 * @param   mem     Memory to attach.  
 * @param   sync    Synchronization mode. 
 * @return          Operation status.
 * @see             kCudaManagedAlloc, kCudaStream_AttachMemory
*/
kFsFx(kStatus) kCudaStream_DetachMemory(kCudaStream stream, kPointer mem, kCudaSync sync = kCUDA_SYNC_DEFAULT);

#endif
