/** 
 * @file    kCudaAlloc.h
 * @brief   Declares the kCudaAlloc class. 
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CUDA_ALLOC_H
#define K_FIRESYNC_CUDA_ALLOC_H

#include <kFireSync/Cuda/kCudaDef.h>
#include <kFireSync/Cuda/kCudaAlloc.x.h>

/**
 * @class   kCudaDeviceAlloc
 * @extends kObject
 * @ingroup kFireSync-Cuda
 * @brief   Represents a Cuda device memory allocator object. 
 */
//typedef kObject kCudaDeviceAlloc;               --forward-declared in kFsDef.x.h

/** 
 * Gets the singleton Cuda device memory allocator instance.
 * 
 * Note, the allocator retured by this method may be wrapped in another allocator; do 
 * not assume that the type of the returned allocator is kCudaDeviceAlloc. 
 *
 * @public      @memberof kCudaDeviceAlloc
 * @return      Memory allocator capable of allocating CUDA device memory. 
 */
kInlineFx(kAlloc) kCudaDeviceAlloc_Instance()
{
    return xkCudaAlloc_Static()->deviceAlloc;
}

/**
 * @class   kCudaManagedAlloc
 * @extends kObject
 * @ingroup kFireSync-Cuda
 * @brief   Represents a Cuda managed memory allocator object. 
 * 
 * For devices that can support concurrent access to managed memory (both device and host can safely 
 * access the memory simultaneously), this allocator will allocate memory with global attachment. There
 * are no special requirements when using this kind of managed memory. 
 * 
 * For devices that cannot support concurrent access to managed memory (device and host cannot safely 
 * access the memory simultaneously), this allocator will allocate memory with host attachment. In 
 * this state, the memory can be accessed safely from any host (CPU) threads but cannot be accessed 
 * by a Cuda device. In order to allow a Cuda device to access this memory, the memory must first 
 * be attached to a stream. Once attached, this memory can only be accessed by host threads or by 
 * the device while executing operations on the specified stream. In this state, it is the caller's 
 * responsibility to ensure that the memory is not accessed simultaneously by the host and the device. 
 * When device operations with this memory have been completed, the memory can be detached from the 
 * stream, once again enabling unrestricted access from host threads. 
 * 
 * Allocator traits can be accessed (via kAlloc_Traits) to determine whether concurrent managed 
 * access is supported. The presence of the kALLOC_TRAIT_SERIAL trait indicates that concurrent managed
 * access is not supported. Alternatively, the kCudaDevice_Properties method can be used to determine  
 * this information directly from device properties. However, if both cases must be supported,
 * then the simplest course of action is to assume that concurrent managed access is not available, and to 
 * perform the necessary attach/detach operations to ensure safe access. For systems with concurrent
 * managed access, these operations will have no effect. 
 */
//typedef kObject kCudaManagedAlloc;               --forward-declared in kFsDef.x.h

/** 
 * Gets the singleton Cuda managed memory allocator instance.
 * 
 * Note, the allocator retured by this method may be wrapped in another allocator; do 
 * not assume that the type of the returned allocator is kCudaManagedAlloc. 
 *
 * @public      @memberof kCudaManagedAlloc
 * @return      Memory allocator capable of allocating CUDA managed memory. 
 */
kInlineFx(kAlloc) kCudaManagedAlloc_Instance()
{
    return xkCudaAlloc_Static()->managedAlloc;
}

/**
 * @class   kCudaPinnedAlloc
 * @extends kObject
 * @ingroup kFireSync-Cuda
 * @brief   Represents a Cuda pinned memory allocator object. 
 */
//typedef kObject kCudaPinnedAlloc;               --forward-declared in kFsDef.x.h

/** 
 * Gets the singleton Cuda pinned memory allocator instance.
 * 
 * Note, the allocator retured by this method may be wrapped in another allocator; do 
 * not assume that the type of the returned allocator is kCudaPinnedAlloc. 
 *
 * @public      @memberof kCudaPinnedAlloc
 * @return      Memory allocator capable of allocating CUDA pinned memory. 
 */
kInlineFx(kAlloc) kCudaPinnedAlloc_Instance()
{
    return xkCudaAlloc_Static()->pinnedAlloc;
}

#endif
