/** 
 * @file    kSystem.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_SYSTEM_X_H
#define K_FIRESYNC_SYSTEM_X_H

#define kSYSTEM_DEFAULT_DATA_QUEUE_CAPACITY     (1 << 25)       //Default system-wide data message queue capacity (bytes).
#define kSYSTEM_DEFAULT_HEALTH_QUEUE_CAPACITY   (1 << 20)       //Default system-wide health message queue capacity (bytes).

#define kSYSTEM_DATA_QUIT_QUERY_PERIOD          (100000)        //Polling interval for quit status (us). 
#define kSYSTEM_HEALTH_QUIT_QUERY_PERIOD        (100000)        //Polling interval for quit status (us). 

#define kSYSTEM_STATUS_CHECK_PERIOD             (1000000)       //Period at which node status checks (reset, activity) are performed. 

#define kSYSTEM_HOST_HEALTH_PERIOD              (1000000)       //Period at which host health is dispatched (if enabled).

typedef struct kNodeFilterCallback
{
    kNodeFilterFx function;       ///< Callback function.
    kPointer receiver;            ///< Callback receiver context pointer. 
} kNodeFilterCallback;

typedef struct kSystemClass
{
    kObjectClass base; 

    kLock stateLock;                         //Protects state accessed from multiple threads. 

    kSize dataNetBufferSize;                 //Net buffer size for node data channels.

    kArrayList discoveryProviders;           //Discovery providers -- kArrayList<kDiscoveryProvider>.
    kArrayList allNodes;                     //All known nodes -- kArrayList<kNode>.
    kArrayList onlineNodes;                  //Nodes disclosed publicly -- kArrayList<kNode>.
    kMap discoveryMap;                       //Maps from node id to latest received discovery information -- kMap<k32u, kDiscoveryInfo>.

    kArrayList errors;                       //Errors detected during previous verification -- kArrayList<kText128>.
    kArrayList warnings;                     //Warnings detected during previous verification -- kArrayList<kText128>.

    kMsgQueue dataQueue;                     //Enqueues received messages if no callback installed -- kMsgQueue<kMsgInfo>.
    kCallback dataFx;                        //Data callback function.

    kMsgQueue healthQueue;                   //Enqueues received messages if no callback installed -- kMsgQueue<kHealthMsg>. 
    kCallback healthFx;                      //Health callback function.
    kHealth hostHealthProvider;              //Optionally provides health information for local host. 
    kPeriodic hostHealthTimer;               //Periodically generates health messages for local host. 
    k32u hostHealthId;                       //Node identifier to use for health messages originating from hostHealthProvider.

    kPeriodic statusTimer;                   //Periodic timer to check for various node status changes. 
    kEvent onStatusCheck;                    //Listeners for status check event. 
 
    kNodeFilterCallback nodeFilterCallback;

    kNetworkInfo netInfo;                    //Network information, used for end point preference comparisons. 
    kAtomic32s netInfoChanged;               //Has network configuration information changed?

} kSystemClass; 

kDeclareClassEx(kFs, kSystem, kObject)

kFsFx(kStatus) kSystem_Init(kSystem system, kType type, kAlloc alloc); 
kFsFx(kStatus) kSystem_VRelease(kSystem system);

kFsFx(kStatus) kSystem_AddDiscoveryEx(kSystem system, kType providerType, kDiscoveryProvider* provider); 

kFsFx(kStatus) kSystem_StartDiscoveryEnum(kSystem system); 
kFsFx(kStatus) kSystem_WaitDiscoveryFirst(kSystem system); 

kFsFx(kStatus) kSystem_OnDiscoveryEnum(kSystem system, kDiscoveryProvider provider, kArrayList infoList); 

kFsFx(kStatus) kSystem_OnNodeAddressChangeRequest(kSystem system, kDiscoveryProvider provider, const kDiscoverySetAddressInfo* info); 

kFsFx(kStatus) kSystem_FindDiscoveryInfo(kSystem system, k32u nodeId, kDiscoveryInfo* info); 

kFsFx(kStatus) kSystem_FindNodeAll(kSystem system, k32u id, kNode* node); 

kFsFx(kStatus) kSystem_RefreshNodeList(kSystem system, kArrayList removed, kArrayList added); 

kFsFx(kStatus) kSystem_AddStatusCheckHandler(kSystem system, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kSystem_RemoveStatusCheckHandler(kSystem system, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kSystem_OnStatusCheck(kSystem system, kPeriodic timer); 

kFsFx(kStatus) kSystem_Lock(kSystem system); 
kFsFx(kStatus) kSystem_Unlock(kSystem system); 

kFsFx(kStatus) kSystem_OnData(kSystem system, kObject sender, kMsgInfo message); 

kFsFx(kStatus) kSystem_OnHealth(kSystem system, kObject sender, kObject info); 

kFsFx(kStatus) kSystem_OnHostHealthTimer(kSystem system, kPeriodic timer); 

//[Deprecated] Replace with kSystem_AddDiscoveryAdapter
kInlineFx(kStatus) kSystem_SetDiscoveryAdapter(kSystem system, const kChar* interfaceName)
{
    return kSystem_AddDiscoveryAdapter(system, interfaceName);
}

kFsFx(kStatus) kSystem_OnNetChange(kSystem system, kPointer sender, kPointer args);
kFsFx(kStatus) kSystem_RefreshNetInfo(kSystem system);

kFsFx(kBool) kSystem_IsReachable(kSystem system, kIpAddress address);

#endif
