/** 
 * @file    kRoute.h
 * @brief   Declares the kRoute class. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ROUTE_H
#define K_FIRESYNC_ROUTE_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kRoute
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for a pipe message route.
 */
//typedef kObject kRoute;        --forward-declared in kFsDef.x.h

/** 
 * Sets end-point information for the route source.
 *
 * @public              @memberof kRoute
 * @param   route       Route object.  
 * @param   source      End-point information for route source. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kRoute_SetSource(kRoute route, const kRoutePoint* source);

/** 
 * Returns end-point information for the route source.
 *
 * @public              @memberof kRoute
 * @param   route       Route object.  
 * @return              Returns end-point information for the route source.
 */
kFsFx(const kRoutePoint*) kRoute_Source(kRoute route);

/** 
 * Sets end-point information for the route destination.
 *
 * @public              @memberof kRoute
 * @param   route       Route object.  
 * @param   destination End-point information for route destination. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kRoute_SetDestination(kRoute route, const kRoutePoint* destination);

/** 
 * Returns end-point information for the route destination.
 *
 * @public              @memberof kRoute
 * @param   route       Route object.  
 * @return              Returns end-point information for the route destination.
 */
kFsFx(const kRoutePoint*) kRoute_Destination(kRoute route);

/** 
 * Returns constraint and validity information for the route.
 *
 * The following violation types are relevant for routes: 
 * - kVIOLATION_UNIQUENESS: The route is a duplicate of another route.
 * - kVIOLATION_EXISTENCE: The source and/or destination end-points do not exist.  
 *
 * @public              @memberof kRoute
 * @param   route       Route object.  
 * @param   info        Destination for info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kRoute_Info(kRoute route, kInfo* info);

#include <kFireSync/Client/kRoute.x.h>

#endif
