/** 
 * @file    kEncoder.h
 * @brief   Declares the kEncoder type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ENCODER_H
#define K_FIRESYNC_ENCODER_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kEncoder
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for a node's local encoder. 
 */
//typedef kObject kEncoder;        --forward-declared in kFsDef.x.h

/** 
 * Resets encoder device settings to defaults. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_ClearAll(kEncoder encoder);

/** 
 * Enables or disables the encoder. 
 *
 * Disabled encoders are not checked during verification.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   enable      Specifies whether to enable or disable the encoder.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_Enable(kEncoder encoder, kBool enable);

/** 
 * Reports whether the encoder is currently enabled. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              Returns whether the encoder is currently enabled. 
 */
kFsFx(kBool) kEncoder_IsEnabled(kEncoder encoder);

/** 
 * Sets the current interpretation of encoder A/B signals to yield forward/reverse direction.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   direction   Direction mode. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SetDirection(kEncoder encoder, kEncoderDirection direction);

/** 
 * Gets the current interpretation of encoder A/B signals to yield forward/reverse direction.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              Direction mode.
 */
kFsFx(kEncoderDirection) kEncoder_Direction(kEncoder encoder);

/** 
 * Reports constraint and validity information for the Direction setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_DirectionInfo(kEncoder encoder, kInfoBits* info);

/** 
 * Enables reset-on-index. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   enabled     kTRUE to enable reset-on-index; kFALSE otherwise. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_EnableIndexReset(kEncoder encoder, kBool enabled);

/** 
 * Reports whether reset-on-index is enabled. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              kTRUE if reset-on-index is enabled; kFALSE otherwise. 
 */
kFsFx(kBool) kEncoder_IndexResetEnabled(kEncoder encoder);

/** 
 * Reports constraint and validity information for the IndexResetEnabled setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_IndexResetEnabledInfo(kEncoder encoder, kInfoBool* info);

/** 
 * Enables encoder simulation. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   enabled     kTRUE to enable reset-on-index; kFALSE otherwise. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_EnableSimulation(kEncoder encoder, kBool enabled);

/** 
 * Reports whether encoder simulation is enabled. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              kTRUE if simulation is enabled; kFALSE otherwise. 
 */
kFsFx(kBool) kEncoder_SimulationEnabled(kEncoder encoder);

/** 
 * Reports constraint and validity information for the SimulationEnabled setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SimulationEnabledInfo(kEncoder encoder, kInfoBool* info);

/** 
 * Sets the encoder simulation direction.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   direction   kTRUE for increasing count; kFALSE for decreasing count.
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SetSimulationDirection(kEncoder encoder, kBool direction);

/** 
 * Reports the simulation direction.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              kTRUE for increasing count; kFALSE for decreasing count.
 */
kFsFx(kBool) kEncoder_SimulationDirection(kEncoder encoder);

/** 
 * Reports constraint and validity information for the SimulationDirection setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SimulationDirectionInfo(kEncoder encoder, kInfoBool* info);

/**
* Sets the simulation period, in nanoseconds.
*
* Period values are specified in FireSync nanoseconds (~ 1 ns).
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @param   period      Simulation period, in nanoseconds.
* @return              Operation status.
*/
kFsFx(kStatus) kEncoder_SetSimulationPeriod(kEncoder encoder, k64u period);

/**
* Gets the simulation period, in nanoseconds.
*
* Simulation period values are specified in FireSync nanoseconds (~ 1 ns).
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @return              Returns the simulation period, in nanoseconds.
*/
kFsFx(k64u) kEncoder_SimulationPeriod(kEncoder encoder);

/**
* Reports constraint and validity information for the SimulationPeriod setting.
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kEncoder_SimulationPeriodInfo(kEncoder encoder, kInfo64u* info);

/**
* Sets the simulation resolution.
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @param   resolution  Simulation resolution
* @return              Operation status.
*/
kFsFx(kStatus) kEncoder_SetSimulationResolution(kEncoder encoder, k32u resolution);

/**
* Gets the simulation resolution.
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @return              Returns the simulation resolution.
*/
kFsFx(k32u) kEncoder_SimulationResolution(kEncoder encoder);

/**
* Reports constraint and validity information for the SimulationResolution setting.
*
* @public              @memberof kEncoder
* @param   encoder     Encoder object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kEncoder_SimulationResolutionInfo(kEncoder encoder, kInfo32u* info);

/** 
 * Sets the snapshot source for an encoder. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   source      Snapshot source.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SetSnapshotSource(kEncoder encoder, kEncoderSnapshotSource source);

/** 
 * Retrieves the snapshot source.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.   
 * @return              Snapshot source. 
 */
kFsFx(kEncoderSnapshotSource) kEncoder_SnapshotSource(kEncoder encoder);

/** 
 * Reports constraint and validity information for the Snapshot Source setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SnapshotSourceInfo(kEncoder encoder, kInfoBits* info);

/** 
 * Sets the snapshot source id for an encoder. 
 *
 * For snapshot source modes that suppport multiple sources, 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   id          Specifies the snapshot source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SetSnapshotSourceId(kEncoder encoder, k32u id);

/** 
 * Gets the snapshot source id for an encoder.
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              Input counter source id.
 */
kFsFx(k32u) kEncoder_SnapshotSourceId(kEncoder encoder);

/** 
 * Reports constraint and validity information for the SnapshotSourceId setting. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_SnapshotSourceIdInfo(kEncoder encoder, kInfo* info);

/** 
 * Resets the counter of this encoder module. 
 * 
 * This function affects the node's local encoder; if the node is receiving synchronization data from a master, 
 * this function has no effect on the reported encoder count. 
 *
 * @public              @memberof kEncoder
 * @param   encoder     Encoder object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kEncoder_Reset(kEncoder encoder);

#include <kFireSync/Client/kEncoder.x.h>

#endif
