/** 
 * @file    kInvertBlockInfo.h
 * @brief   Declares the kInvertBlockInfo class. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_INVERT_BLOCK_INFO_H
#define K_FIRESYNC_INVERT_BLOCK_INFO_H

#include <kFireSync/kNodeDef.h>

/**
 * @struct  kInvertBlockAlgorithm
 * @extends kValue
 * @ingroup kFireSync-Blocks
 * @brief   Represents a kInvertBlock algorithm type. 
 */
typedef k32s kInvertBlockAlgorithm; 

/** @relates kInvertBlockAlgorithm @{ */
#define kINVERT_BLOCK_ALGORITHM_HOST            (0x1)       ///< Standard host implementation.
#define kINVERT_BLOCK_ALGORITHM_CUDA            (0x2)       ///< Cuda implementation.
/** @} */

/**
 * @struct  kInvertBlockMemory
 * @extends kValue
 * @ingroup kFireSync-Blocks
 * @brief   Represents a kInvertBlock memory type. 
 */
typedef k32s kInvertBlockMemory; 

/** @relates kInvertBlockMemory @{ */
#define kINVERT_BLOCK_MEMORY_HOST               (0x1)       ///< Standard host memory.
#define kINVERT_BLOCK_MEMORY_CUDA               (0x2)       ///< Any Cuda device accessible memory.
#define kINVERT_BLOCK_MEMORY_CUDA_DEVICE        (0x4)       ///< Cuda device memory.
#define kINVERT_BLOCK_MEMORY_CUDA_MANAGED       (0x8)       ///< Cuda managed memory.
/** @} */

#include <kFireSync/Blocks/kInvertBlockInfo.x.h>

/**
 * @class   kInvertBlockInfo
 * @extends kObject
 * @ingroup kFireSync-Blocks
 * @brief   Represents configuration for kInvertBlock. 
 */
//typedef kObject kInvertBlockInfo;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kInvertBlockInfo object. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Receives constructed info object. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
 kFsFx(kStatus) kInvertBlockInfo_Construct(kInvertBlockInfo* info, kAlloc allocator); 

/** 
 * Constructs a kInvertBlockInfo object and loads its content from an XML object. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Receives constructed info object.  
 * @param   xml         Source from which parameters are loaded. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Load(kInvertBlockInfo* info, kXml xml, kAlloc allocator); 

/** 
 * Constructs a kXml object and uses it to store the content of this kInvertBlockInfo instance.
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   xml         Receives constructed kXml object.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Store(kInvertBlockInfo info, kXml* xml, kAlloc allocator); 

/** 
 * Reads kInvertBlockInfo state from a kXml object. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Read(kInvertBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Writes kInvertBlockInfo state to a kXml object. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Write(kInvertBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Checks and automatically corrects all state variables. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Validate(kInvertBlockInfo info); 

/** 
 * Resets all state variables to default values.
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertBlockInfo_Clear(kInvertBlockInfo info); 

/** 
 * Enables the block. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   enabled     kTRUE to enable.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kInvertBlockInfo_Enable(kInvertBlockInfo info, kBool enabled)
{
    kObj(kInvertBlockInfo, info); 

    obj->enabled = enabled;

    return kOK; 
}

/** 
 * Reports whether the block is enabled. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @return              kTRUE if enabled.
 */
kInlineFx(kBool) kInvertBlockInfo_IsEnabled(kInvertBlockInfo info)
{
    kObj(kInvertBlockInfo, info); 

    return obj->enabled; 
}

/** 
 * Sets the algorithm implementation that should be used. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   algorithm   Algorithm implementation.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kInvertBlockInfo_SetAlgorithm(kInvertBlockInfo info, kInvertBlockAlgorithm algorithm)
{
    kObj(kInvertBlockInfo, info); 

    obj->algorithm = algorithm;

    return kOK; 
}

/** 
 * Reports the selected algorithm implementation. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @return              Algorithm implementation. 
 */
kInlineFx(kInvertBlockAlgorithm) kInvertBlockInfo_Algorithm(kInvertBlockInfo info)
{
    kObj(kInvertBlockInfo, info); 

    return obj->algorithm;
}

/** 
 * Sets the output memory type.
 * 
 * This setting is only applicable to specific algorithm types (e.g., Cuda).
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @param   memory      Memory type.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kInvertBlockInfo_SetOutputMemory(kInvertBlockInfo info, kInvertBlockMemory memory)
{
    kObj(kInvertBlockInfo, info); 

    obj->memory = memory;

    return kOK; 
}

/** 
 * Reports the selected output memory type. 
 *
 * @public              @memberof kInvertBlockInfo
 * @param   info        Info object.  
 * @return              Memory type.
 */
kInlineFx(kInvertBlockMemory) kInvertBlockInfo_OutputMemory(kInvertBlockInfo info)
{
    kObj(kInvertBlockInfo, info); 

    return obj->memory;
}

/** Gets the input port id. */
kInlineFx(k16u) kInvertBlockInfo_InputPortId() { return 0;}

/** Gets the output port id. */
kInlineFx(k16u) kInvertBlockInfo_OutputPortId() { return 1; }

#endif
