#ifndef GV_DATA_MSG_X_H
#define GV_DATA_MSG_X_H

#include <GoVision/Data/GvDataMsg.h>
#include <GoVision/GvUtils.h>
#include <kApi/kApiDef.h>
#include <kApi/Data/kMath.h>
#include <kApi/Data/kString.h>
#include <kApi/Io/kDat6Serializer.h>
#include <kApi/Threads/kLock.h>
#include <math.h>

#define GV_DATA_MSG_ALIGNMENT_BYTES         (8)
#define GV_DATA_MSG_POSE_CAPACITY           (5)
#define GV_DATA_MSG_LAYER_CAPACITY          (4)

#define GvDataMsg_PadSize_(SIZE, ALIGNMENT) (((SIZE) + (ALIGNMENT) - 1) / (ALIGNMENT) * (ALIGNMENT))

typedef struct GvDataMsgPose
{
    k32s id;
    kPose2d64f pose;
} GvDataMsgPose;

typedef struct GvDataMsgProfilePose
{
    k32s id;
    kPose2d64f pose;
} GvDataMsgProfilePose;

typedef struct GvDataMsgFrameClass
{
    GvDataMsg parent;
    kSize index;
    kStamp stamp;
    k64u time;
    k64s encoder;
    k64u ptpTime;
    GvMovementDirection direction;
    kPoint3d64f offset;
    kPoint3d64f scale;
    kSize poseCount;
    GvDataMsgPose poses[GV_DATA_MSG_POSE_CAPACITY];
    kObject objectExtension;
    kObject renderingObject;
} GvDataMsgFrameClass;

#define GvDataMsgFrame_Cast_(S)         kCast(GvDataMsgFrameClass*, S)

typedef struct GvDataMsgLayer
{
    k32s id;
    kType type;
    kSize frameSize;
    kSize frameStride;
    kSize bufferSize;
    kBool allocated;
    void* buffer;
    kArray1 holderArrays;
} GvDataMsgLayer;

typedef struct GvDataMsgReadHeader
{
    k32s disposition;
    kSize length;
    kSize width;
    kSize count;
    kSize layerCount;
    kType stampType;
    kTypeVersion stampVersion;
} GvDataMsgReadHeader;

typedef struct GvDataMsgStatic
{
    kLock holderLock;
} GvDataMsgStatic;

typedef struct GvDataMsgVTable
{
    kMsgSetVTable base;

    kPoint3d64f (kCall* VOffsetWithRef)(GvDataMsg msg); 
} GvDataMsgVTable;

typedef struct GvDataMsgClass
{
    kMsgSetClass base;
    GvDataMsgDisposition disposition;
    kSize count;
    kSize length;
    kSize width;
    kSize layerCount;

    kSize bufferSize;
    GvDataMsgFrameClass* buffer;
    GvDataMsgLayer layers[GV_DATA_MSG_LAYER_CAPACITY];
} GvDataMsgClass;

kDeclareFullClassEx(Gv, GvDataMsg, kMsgSet)

GvFx(kStatus) xGvDataMsg_InitStatic();
GvFx(kStatus) xGvDataMsg_ReleaseStatic();

GvFx(kStatus) GvDataMsg_Init(GvDataMsg msg, kType type, GvDataMsgDisposition disposition, kSize count, kType pointType, kSize width, kSize length, kAlloc alloc);
GvFx(kStatus) GvDataMsg_VRelease(GvDataMsg msg);
GvFx(kStatus) GvDataMsg_VInitClone(GvDataMsg msg, GvDataMsg src, kAlloc alloc);

GvFx(kSize) GvDataMsg_VSize(GvDataMsg msg);

GvFx(kSize) GvDataMsg_VCount(GvDataMsg msg);
GvFx(const kStamp*) GvDataMsg_VStampAt(GvDataMsg msg, kSize frame);
GvFx(kStatus) GvDataMsg_VSetStampAt(GvDataMsg msg, kSize frame, const kStamp* stamp);
GvFx(kStatus) GvDataMsg_VDataAt(GvDataMsg msg, kSize frame, kArrayList* list);

GvFx(kPoint3d64f) GvDataMsg_VOffsetWithRef(GvDataMsg msg);

GvFx(kStatus) GvDataMsg_Assign(GvDataMsg msg, GvDataMsg src);

GvFx(kStatus) GvDataMsg_WriteDat6V0(GvDataMsg msg, kSerializer serializer);
GvFx(kStatus) GvDataMsg_ReadDat6V0(GvDataMsg msg, kSerializer serializer, kAlloc allocator);

GvFx(kStatus) GvDataMsg_WriteDat6V1(GvDataMsg msg, kSerializer serializer);
GvFx(kStatus) GvDataMsg_WriteMsg(GvDataMsg msg, kSerializer serializer);
GvFx(kStatus) GvDataMsg_WriteMsgHeader(GvDataMsg msg, kSerializer serializer, kTypeVersion* stampVersion);
GvFx(kStatus) GvDataMsg_WriteMsgAttrs(GvDataMsg msg, kSerializer serializer, kTypeVersion stampVersion);
GvFx(kStatus) GvDataMsg_WriteMsgLayers(GvDataMsg msg, kSerializer serializer);

GvFx(kStatus) GvDataMsg_ReadDat6V1(GvDataMsg msg, kSerializer serializer, kAlloc allocator);
GvFx(kStatus) GvDataMsg_ReadMsgHeader(GvDataMsg msg, kSerializer serializer, GvDataMsgReadHeader* header);
GvFx(kStatus) GvDataMsg_ReadMsgContents(GvDataMsg msg, kSerializer serializer, const GvDataMsgReadHeader* header);
GvFx(kStatus) GvDataMsg_ReadMsgAttrs(GvDataMsg msg, kSerializer serializer, kType stampType, kTypeVersion stampVersion);
GvFx(kStatus) GvDataMsg_ReadMsgAttrsFrame(GvDataMsg msg, kSerializer serializer, kType stampType, kTypeVersion stampVersion, GvDataMsgFrameClass* frame);
GvFx(kStatus) GvDataMsg_ReadMsgLayers(GvDataMsg msg, kSerializer serializer, kSize layerCount);
GvFx(kStatus) GvDataMsg_ReadMsgLayerAttr(GvDataMsg msg, kSerializer serializer, k32s* layerId, kType* layerType, kTypeVersion* typeVersion);

GvFx(kStatus) GvDataMsg_ReadDat6ProfileV0(GvDataMsg msg, kSerializer serializer, kAlloc allocator);
GvFx(kStatus) GvDataMsg_ReadDat6ProfileV1(GvDataMsg msg, kSerializer serializer, kAlloc allocator);
GvFx(kStatus) GvDataMsg_WriteDat6Profile(GvDataMsg msg, kSerializer serializer);
GvFx(kStatus) GvDataMsg_ReadDat6Profile(GvDataMsg msg, kSerializer serializer, kAlloc allocator, k32u version);

GvFx(kStatus) GvDataMsg_Allocate(GvDataMsg msg, kType pointType);
GvFx(kStatus) GvDataMsg_InitLayer(GvDataMsg msg, GvDataMsgLayer* layer, k32s id, kType type, kBool allocate);
GvFx(GvDataMsgLayer*) GvDataMsg_FindLayer(GvDataMsg msg, k32s id);
GvFx(kStatus) GvDataMsg_SetLayerDataImpl(GvDataMsg msg, GvDataMsgLayer* layer, kSize frameIndex, const void* data);
GvFx(kStatus) GvDataMsg_SetLayerDataImplPartial(GvDataMsg msg, GvDataMsgLayer* layer, kSize frameIndex, kSize offset, kSize length, const void* data);
GvFx(kStatus) GvDataMsg_AttachLayerDataImpl(GvDataMsg msg, GvDataMsgLayer* layer, kSize frameIndex, void* dataPtr);
GvFx(void*) GvDataMsg_LayerDataImpl(GvDataMsg msg, GvDataMsgLayer* layer, kSize frameIndex);
GvFx(kStatus) GvDataMsg_LayerDataArrayImpl(GvDataMsg msg, GvDataMsgLayer* layer, kSize frameIndex, kObject* dataHolder);

GvFx(kObject) GvDataMsg_RangeArray(GvDataMsg msg);
GvFx(kObject) GvDataMsg_LayerArray(GvDataMsg msg, k32s id);

// Batching-compliant

GvFx(GvDataMsgFrame) GvDataMsg_FrameAt(GvDataMsg msg, kSize index);

GvFx(kStatus) GvDataMsgFrame_CopyAttrs(GvDataMsgFrame frame, GvDataMsgFrame source);
GvFx(kStatus) GvDataMsgFrame_Copy(GvDataMsgFrame frame, GvDataMsgFrame source);

GvFx(GvDataMsg) GvDataMsgFrame_Parent(GvDataMsgFrame frame);

GvFx(kSize) GvDataMsgFrame_Width(GvDataMsgFrame frame);
GvFx(kSize) GvDataMsgFrame_Length(GvDataMsgFrame frame);

GvFx(kStatus) GvDataMsgFrame_SetRanges(GvDataMsgFrame frame, const void* data);
GvFx(void*) GvDataMsgFrame_Ranges(GvDataMsgFrame frame, kSize row, kSize column);
GvFx(kObject) GvDataMsgFrame_RangeArray(GvDataMsgFrame frame);

GvFx(kStatus) GvDataMsgFrame_SetLayerData(GvDataMsgFrame frame, k32s id, const void* data);
GvFx(kStatus) GvDataMsgFrame_SetLayerDataPartial(GvDataMsgFrame frame, k32s id, kSize offset, kSize length, const void* data);
GvFx(kStatus) GvDataMsgFrame_AttachLayerData(GvDataMsgFrame frame, k32s id, void* data);
GvFx(void*) GvDataMsgFrame_LayerData(GvDataMsgFrame frame, k32s id, kSize row, kSize column);
GvFx(kObject) GvDataMsgFrame_LayerArray(GvDataMsgFrame frame, k32s id);

GvFx(k64u) GvDataMsgFrame_Time(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetTime(GvDataMsgFrame frame, k64u time);

GvFx(kStatus) GvDataMsgFrame_SetFsAndPtpTime(GvDataMsgFrame frame, k64u time, k64u ptpTime);

GvFx(k64u) GvDataMsgFrame_PtpTime(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetPtpTime(GvDataMsgFrame frame, k64u time);

GvFx(k64s) GvDataMsgFrame_Encoder(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetEncoder(GvDataMsgFrame frame, k64s encoder);

GvFx(const kPoint3d64f*) GvDataMsgFrame_Offset(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetOffset(GvDataMsgFrame frame, const kPoint3d64f* offset);

GvFx(const kPoint3d64f*) GvDataMsgFrame_Scale(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetScale(GvDataMsgFrame frame, const kPoint3d64f* scale);

GvFx(GvMovementDirection) GvDataMsgFrame_Direction(GvDataMsgFrame frame);
GvFx(kStatus) GvDataMsgFrame_SetDirection(GvDataMsgFrame frame, GvMovementDirection direction);

GvFx(kStatus) GvDataMsgFrame_AddPose(GvDataMsgFrame frame, k32s id, const kPose2d64f* pose);
GvFx(kStatus) GvDataMsgFrame_ClearPose(GvDataMsgFrame frame);

GvFx(kSize) GvDataMsgFrame_PoseCount(GvDataMsgFrame frame);
GvFx(k32s) GvDataMsgFrame_PoseIdAt(GvDataMsgFrame frame, kSize index);
GvFx(const kPose2d64f*) GvDataMsgFrame_PoseAt(GvDataMsgFrame frame, kSize index);

GvFx(kStatus) GvDataMsgFrame_SetPoseIdAt(GvDataMsgFrame frame, kSize index, k32s id);
GvFx(kStatus) GvDataMsgFrame_SetPoseAt(GvDataMsgFrame frame, kSize index, const kPose2d64f* pose);

GvFx(kStatus) GvDataMsgFrame_CopyPose(GvDataMsgFrame frame, GvDataMsgFrame source);
GvFx(kStatus) GvDataMsgFrame_CopyPoseToId(GvDataMsgFrame frame, GvDataMsgFrame source, k32s id);

GvFx(kStatus) GvDataMsgFrame_SelectPose(GvDataMsgFrame frame, k32s toId, kPose2d64f* pose);
GvFx(kStatus) GvDataMsgFrame_SelectPoseInverse(GvDataMsgFrame frame, k32s fromId, kPose2d64f* pose);
GvFx(kStatus) GvDataMsgFrame_SelectPoseRelative(GvDataMsgFrame frame, k32s toId, k32s fromId, kPose2d64f* pose);

GvFx(kStamp*) GvDataMsgFrame_Stamp(GvDataMsgFrame frame);

#endif
