/**
* @file    GvDataMsg.h
* @brief   Declares the GvDataMsg class.
*
* Copyright (C) 2019-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GV_DATA_MSG_H
#define GV_DATA_MSG_H

#include <GoVision/GvDef.h>
#include <GoVision/GvUtils.h>
#include <kFireSync/Data/kMsgSet.h>
#include <kApi/Data/kArray1.h>
#include <kApi/Data/kArrayList.h>

/**
* Base class for data messages.
*
* @class       GvDataMsg
* @extends     kMsgSet
* @ingroup     GoVision-Data
*/
typedef kMsgSet GvDataMsg;

typedef kPointer GvDataMsgFrame;

typedef k32s GvDataMsgDisposition;

#define GV_DATA_MSG_DISPOSITION_NONE    (0)
#define GV_DATA_MSG_DISPOSITION_RANGE   (1)
#define GV_DATA_MSG_DISPOSITION_PROFILE (2)
#define GV_DATA_MSG_DISPOSITION_SURFACE (3)
#define GV_DATA_MSG_DISPOSITION_GENERIC (10)

#define GV_DATA_MSG_LAYER_RANGE         (0)
#define GV_DATA_MSG_LAYER_INTENSITY     (1)
#define GV_DATA_MSG_LAYER_MAP           (2)
#define GV_DATA_MSG_LAYER_SLICES        (3)

GvFx(kStatus) GvDataMsg_Construct(GvDataMsg* msg, GvDataMsgDisposition disposition, kType pointType, kSize count, kSize width, kSize length, kAlloc alloc);

/**
 * Convenience wrapper that copies elementary data attributes from the source for first frame.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   source      Source message object.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_CopyAttrs(GvDataMsg msg, GvDataMsg source);

/**
 * Returns the message disposition type.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Disposition type.
 */
GvFx(GvDataMsgDisposition) GvDataMsg_Disposition(GvDataMsg msg);

/**
 * Returns the message range data type.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Range data type.
 */
GvFx(kType) GvDataMsg_RangeType(GvDataMsg msg);

/**
 * Returns the data width as a number of columns.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Column count.
 */

GvFx(kSize) GvDataMsg_Width(GvDataMsg msg);

/**
 * Returns the data length as a number of rows.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Row count.
 */
GvFx(kSize) GvDataMsg_Length(GvDataMsg msg);

/**
 * Sets range layer data.
 * NOTE: This copies the data.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   data        Pointer to range layer data.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetRanges(GvDataMsg msg, const void* data);

/**
 * Returns a pointer to range layer data at given row and column.
 * NOTE: A range layer should be present.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   row         Range data row.
 * @param   column      Range data column.
 * @return              Pointer to range layer data.
 */
GvFx(void*) GvDataMsg_Ranges(GvDataMsg msg, kSize row, kSize column);

/**
 * Adds a new layer to the end (and allocates it).
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   type        Layer data type.
 * @param   id          Layer id.
 * @return              Operation Status.
 */
GvFx(kStatus) GvDataMsg_AddLayer(GvDataMsg msg, kType type, k32s id);

/**
 * Appends a layer to the end (optionally allocating it).
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   type        Layer data type.
 * @param   id          Layer id.
 * @param   allocate    True to allocate the layer.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_AppendLayer(GvDataMsg msg, kType type, k32s id, kBool allocate);

/**
 * Returns a layer count.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Layer count.
 */
GvFx(kSize) GvDataMsg_LayerCount(GvDataMsg msg);

/**
 * Returns the layer id at the given layer index.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   index       Layer index.
 * @return              Layer id.
 */
GvFx(k32s) GvDataMsg_LayerIdAt(GvDataMsg msg, kSize index);

/**
 * Returns whether or not this message contains the given layer id.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   id          Layer id.
 * @return              Presence of layer id.
 */
GvFx(kBool) GvDataMsg_HasLayer(GvDataMsg msg, k32s id);

/**
 * Returns the layer data type of the given layer id.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   id          Layer id.
 * @return              Layer data type.
 */
GvFx(kType) GvDataMsg_LayerType(GvDataMsg msg, k32s id);

/**
 * Sets the layer data.
 * NOTE: This copies the data.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   id          Layer id.
 * @param   data        Pointer to layer data.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetLayerData(GvDataMsg msg, k32s id, const void* data);

/**
 * Gets the layer data at the given row and column.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   id          Layer id.
 * @param   row         Layer data row.
 * @param   column      Layer data column.
 * @return              Pointer to layer data.
 */
GvFx(void*) GvDataMsg_LayerData(GvDataMsg msg, k32s id, kSize row, kSize column);

/**
 * Returns the movement direction.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Movement direction.
 */
GvFx(GvMovementDirection) GvDataMsg_Direction(GvDataMsg msg);

/**
 * Sets the movement direction.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   direction   Movement direction.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetDirection(GvDataMsg msg, GvMovementDirection direction);

/**
 * Returns the data stamp.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Pointer to data stamp.
 */
GvFx(const kStamp*) GvDataMsg_Stamp(GvDataMsg msg);

/**
 * Sets the data stamp.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   stamp       Pointer to data stamp.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetStamp(GvDataMsg msg, const kStamp* stamp);

/**
 * Returns the data time, which is expresseed in Firesync time units in microseconds.
 * To convert Firesync time units to milliseconds, divide by 1024 (not 1000).
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Data time.
 */
GvFx(k64u) GvDataMsg_Time(GvDataMsg msg);

/**
 * Sets the data time (usec).
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   time        Data time.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetTime(GvDataMsg msg, k64u time);

/**
 * Sets the FsTime and PtpTime in (usec).
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   time        Data time.
 * @param   ptpTime     PTP time.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetFsAndPtpTime(GvDataMsg msg, k64u time, k64u ptpTime);

/**
 * Returns the data encoder value.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Data encoder value.
 */
GvFx(k64s) GvDataMsg_Encoder(GvDataMsg msg);

/**
 * Sets the data encoder value.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   encoder     Data encoder value.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetEncoder(GvDataMsg msg, k64s encoder);

/**
 * Returns the ptp timestamp value.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Ptp timestamp value.
 */
GvFx(k64u) GvDataMsg_PtpTime(GvDataMsg msg);

/**
 * Sets the data ptp timestamp value.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   ptpTime     Ptp timestamp value.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetPtpTime(GvDataMsg msg, k64u ptpTime);

/**
 * Returns the x,y,z offset for the data message.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Pointer to data offset.
 */
GvFx(const kPoint3d64f*) GvDataMsg_Offset(GvDataMsg msg);

/**
 * Sets the x,y,z offset for the data message.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   offset      Pointer to data offset.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetOffset(GvDataMsg msg, const kPoint3d64f* offset);

/**
 * Returns the x,y,z scale of the data message.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Pointer to data scale.
 */
GvFx(const kPoint3d64f*) GvDataMsg_Scale(GvDataMsg msg);

/**
 * Sets the x,y,z scale of the data message.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   scale       Pointer to data scale.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetScale(GvDataMsg msg, const kPoint3d64f* scale);

/**
 * Adds pose.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   id          Pose id.
 * @param   pose        Pointer to pose.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_AddPose(GvDataMsg msg, k32s id, const kPose2d64f* pose);

/**
 * Clears all poses.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_ClearPose(GvDataMsg msg);

/**
 * Returns the number of poses.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Pose count.
 */
GvFx(kSize) GvDataMsg_PoseCount(GvDataMsg msg);

/**
 * Returns the pose id for the given pose index.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   index       Pose index.
 * @return              Pose id.
 */
GvFx(k32s) GvDataMsg_PoseIdAt(GvDataMsg msg, kSize index);

/**
 * Returns the pose for the given pose index.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   index       Pose index.
 * @return              Pointer to pose.
 */
GvFx(const kPose2d64f*) GvDataMsg_PoseAt(GvDataMsg msg, kSize index);

/**
 * Sets the pose id for the pose at the given index.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   index       Pose index.
 * @param   id          Pose id.
 * @return              Operation status.
 */

GvFx(kStatus) GvDataMsg_SetPoseIdAt(GvDataMsg msg, kSize index, k32s id);

/**
 * Sets the pose at the given index.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   index       Pose index.
 * @param   pose        Pointer to pose.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetPoseAt(GvDataMsg msg, kSize index, const kPose2d64f* pose);

/**
 * Copies the pose from the source message.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   source      Source message object.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_CopyPose(GvDataMsg msg, GvDataMsg source);

/**
 * Copies all poses less than the id from the source.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   source      Source message object.
 * @param   id          Pose id copy limit (non-inclusive).
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_CopyPoseToId(GvDataMsg msg, GvDataMsg source, k32s id);

/**
 * Selects poses for the given id.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   toId        Pose 'to' selection id (inclusive).
 * @param   pose        Pose selection result.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SelectPose(GvDataMsg msg, k32s toId, kPose2d64f* pose);

/**
 * Selects poses for the given id and provides the inverse.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   fromId      Pose 'from' selection id (inclusive).
 * @param   pose        Pose selection result.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SelectPoseInverse(GvDataMsg msg, k32s fromId, kPose2d64f* pose);

/**
 * Selects the relative pose to get from one pose id to another pose id.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   toId        Pose 'to' selection id (inclusive).
 * @param   fromId      Pose 'from' selection id (inclusive).
 * @param   pose        Pose selection result.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SelectPoseRelative(GvDataMsg msg, k32s toId, k32s fromId, kPose2d64f* pose);

/**
 * Sets the extension object for the given message frame.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   frame       Frame object.
 * @param   object      Extension object.
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetObjectExtAt(GvDataMsg msg, kSize frame, kObject object);

/**
 * Returns the extension object for the given message frame.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   frame       Frame object.
 * @return              Extension object.
 */
GvFx(kObject) GvDataMsg_ObjectExtAt(GvDataMsg msg, kSize frame);

/**
 * Sets the rendering object for the given message frame.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   frame       Frame object.
 * @param   object      Rendering object (GdkGraphic until Studio migrates to newer primitives).
 * @return              Operation status.
 */
GvFx(kStatus) GvDataMsg_SetRenderingAt(GvDataMsg msg, kSize frame, kObject object);

/**
 * Returns the rendering object for the given message frame.
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @param   frame       Frame object.
 * @return              Rendering object (GdkGraphic until Studio migrates to newer primitives).
 */
GvFx(kObject) GvDataMsg_RenderingAt(GvDataMsg msg, kSize frame);

/**
 * Returns the x,y,z offset with respect to frame of reference for the data message .
 *
 * @public              @memberof GvDataMsg
 * @param   msg         Message object.
 * @return              Data offset.
 */
GvFx(kPoint3d64f) GvDataMsg_OffsetWithRef(GvDataMsg msg);

#include <GoVision/Data/GvDataMsg.x.h>

#endif
