/** 
 * @file    TestSurfaceSelect.c
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestSurfaceSelect.h>

kBeginClassEx(Tool, TestSurfaceSelect)
    kAddVMethod(TestSurfaceSelect, kObject, VRelease)
    kAddVMethod(TestSurfaceSelect, GdkTool, VInit)
    kAddVMethod(TestSurfaceSelect, GdkTool, VDescribe)
    kAddVMethod(TestSurfaceSelect, GdkTool, VUpdateConfig)
    kAddVMethod(TestSurfaceSelect, GdkTool, VStart)
    kAddVMethod(TestSurfaceSelect, GdkTool, VStop)
    kAddVMethod(TestSurfaceSelect, GdkTool, VProcess)
kEndClassEx()

ToolFx(kStatus) TestSurfaceSelect_VDescribe(GdkToolInfo toolInfo)
{
    GdkMeasurementInfo mmtInfo;
    GdkRegion3d64f refRegion = { -4, -3, -2, 2, 2, 4, 0 };
    kBool refBool = kTRUE;
    k32s refInt = 0;

    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_SURFACE_SELECT_TOOL_NAME));
    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_SURFACE_SELECT_TOOL_LABEL));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));

    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Y, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_BOOL, "UseRegion", "Use Region", &refBool, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION, "Region", "Region", &refRegion, kNULL));
    
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_INT, "IndexX", "Index X", &refInt, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_INT, "IndexY", "Index Y", &refInt, kNULL));
    
    // Create measurement. Must the same order as measurement index!
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "X", "Position X", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_X));

    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Y", "Position Y", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Y));
      
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Z", "Position Z", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z));

    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_VInit(TestSurfaceSelect tool, kType type, kAlloc alloc)
{
    kObjR(TestSurfaceSelect, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    obj->useRegion = kFALSE;
    kZero(obj->region);
    obj->indexX = 0;
    obj->indexY = 0;

    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_VRelease(TestSurfaceSelect tool)
{
    kObj(TestSurfaceSelect, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestSurfaceSelect_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    GdkParams params = GdkToolCfg_Parameters(toolConfig);
    GdkParam paramUseRegion = GdkParams_Find(params, "UseRegion");
    GdkParam paramRegion = GdkParams_Find(params, "Region");

    kCheck(GdkParam_SetUsed(paramRegion, GdkParam_AsBool(paramUseRegion)));
    
    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_VStart(TestSurfaceSelect tool)
{
    kObj(TestSurfaceSelect, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkParam param;

    obj->dataSource = GdkToolCfg_Source(config);
    
    param = GdkParams_Find(params, "UseRegion");
    obj->useRegion = GdkParam_AsBool(param);

    param = GdkParams_Find(params, "Region");
    obj->region = *GdkParam_AsSurfaceRegion(param);

    param = GdkParams_Find(params, "IndexX");
    obj->indexX = GdkParam_AsInt(param);

    param = GdkParams_Find(params, "IndexY");
    obj->indexY = GdkParam_AsInt(param);

    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_VStop(TestSurfaceSelect tool)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_OutputPoint(TestSurfaceSelect tool, k32u measurementIndex, const kPoint3d64f *point, GdkToolOutput output)
{
    kObj(TestSurfaceSelect, tool);
    GdkGraphic graphic = kNULL;
    GdkGraphicPointSet pointSet = kNULL;
    kStatus exception = kOK;
    kPoint3d32f point32f;
    kAlloc alloc;

    alloc = GdkTool_MessageAlloc(tool);

    // Convert 64f point to 32f. Graphics only supprot 32f data
    if (point->x != k64F_NULL) point32f.x = (k32f)point->x;
    else point32f.x = k32F_NULL;
    
    if (point->y != k64F_NULL) point32f.y = (k32f)point->y;
    else point32f.y = k32F_NULL;

    if (point->z != k64F_NULL) point32f.z = (k32f)point->z;
    else point32f.z = k32F_NULL;

    kTry
    {
        kTest(GdkGraphic_Construct(&graphic, alloc));
        kTest(GdkGraphicPointSet_Construct(&pointSet, 4.0, kMARKER_SHAPE_CROSS, kCOLOR_LIME, &point32f, 1, alloc));
        kTest(GdkGraphic_AddPointSet(graphic, pointSet));
        pointSet = kNULL;

        kTest(GdkToolOutput_SetRendering(output, measurementIndex, graphic));
        graphic = kNULL;
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kDestroyRef(&pointSet);
        kEndFinallyEx();
    }
    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_VProcess(TestSurfaceSelect tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestSurfaceSelect, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkInputItem item;
    GdkDataInfo itemInfo;
    const kPoint3d64f* anchor = GdkToolInput_AnchorPosition(input);
    GdkRegion3d64f offsetRegion = { k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL };
    kStatus exception = kOK;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item) return kERROR_PARAMETER;

    itemInfo = GdkInputItem_Info(item);

    kTry
    {
        k32u regionStartIndexX, regionEndIndexX;
        k32u regionStartIndexY, regionEndIndexY;
        k32u dataIndexX, dataIndexY;

        if(obj->useRegion)
        {
            // Adjust user region based on anchoring information. If anchoring is not enabled, anchor is zero.
            offsetRegion = obj->region;
            offsetRegion.x += anchor->x;
            offsetRegion.y += anchor->y;
            offsetRegion.z += anchor->z;
        }
        else
        {
            // If user didn't select the region, use the maximum region from the input data. 
            offsetRegion.x = GdkInputItem_Offset(item)->x;
            offsetRegion.width = GdkSurfaceInput_ColumnCount(item) * GdkDataInfo_Scale(itemInfo)->x;
            offsetRegion.y = GdkInputItem_Offset(item)->y;
            offsetRegion.length = GdkSurfaceInput_RowCount(item) * GdkDataInfo_Scale(itemInfo)->y;
            offsetRegion.z = GdkDataInfo_Region(itemInfo)->z;
            offsetRegion.height = GdkDataInfo_Region(itemInfo)->height;
        }

        regionStartIndexX = (k32s)((offsetRegion.x - GdkInputItem_Offset(item)->x) / GdkDataInfo_Scale(itemInfo)->x); // index where the region starts 
        regionEndIndexX = (k32s)(((offsetRegion.x + offsetRegion.width) - GdkInputItem_Offset(item)->x) / GdkDataInfo_Scale(itemInfo)->x); // index where the region ends
        
        regionStartIndexY = (k32s)((offsetRegion.y - GdkInputItem_Offset(item)->y) / GdkDataInfo_Scale(itemInfo)->y); // index where the region starts 
        regionEndIndexY = (k32s)(((offsetRegion.y + offsetRegion.length) - GdkInputItem_Offset(item)->y) / GdkDataInfo_Scale(itemInfo)->y); // index where the region ends

        dataIndexX = obj->indexX + regionStartIndexX;
        dataIndexY = obj->indexY + regionStartIndexY;

        if ((dataIndexX > GdkSurfaceInput_ColumnCount(item)) || (obj->indexX > (regionEndIndexX - regionStartIndexX + 1)) ||
            ((dataIndexY > GdkSurfaceInput_RowCount(item)) || (obj->indexY > (regionEndIndexY - regionStartIndexY + 1))))
        {
            kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_X, 0.0, GDK_MEASUREMENT_ERROR_VALUE, config));
            kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_Y, 0.0, GDK_MEASUREMENT_ERROR_VALUE, config));
            kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_Z, 0.0, GDK_MEASUREMENT_ERROR_VALUE, config));
        }
        else
        {
            kPoint3d64f point;
            const k16s *surfaceData = GdkSurfaceInput_RangeRowAt(item, dataIndexY) + dataIndexX;

            point.x = dataIndexX * GdkDataInfo_Scale(itemInfo)->x + GdkInputItem_Offset(item)->x;
            point.y = dataIndexY * GdkDataInfo_Scale(itemInfo)->y + GdkInputItem_Offset(item)->y;

            kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_X, point.x, GDK_MEASUREMENT_OK, config));
            kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_Y, point.y, GDK_MEASUREMENT_OK, config));

            if (*surfaceData != k16S_NULL)
            {
                point.z = (*surfaceData) * GdkDataInfo_Scale(itemInfo)->z + GdkInputItem_Offset(item)->z;                
                kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_Z, point.z, GDK_MEASUREMENT_OK, config));
                kTest(TestSurfaceSelect_OutputPoint(tool, TEST_SURFACE_SELECT_MEASUREMENT_X, &point, output));
                kTest(TestSurfaceSelect_OutputPoint(tool, TEST_SURFACE_SELECT_MEASUREMENT_Y, &point, output));
                kTest(TestSurfaceSelect_OutputPoint(tool, TEST_SURFACE_SELECT_MEASUREMENT_Z, &point, output));
            }                
            else 
            {
                // No z data at that point
                point.z = k32F_NULL;
                kTest(TestSurfaceSelect_OutputValue(output, TEST_SURFACE_SELECT_MEASUREMENT_Z, k64F_NULL, GDK_MEASUREMENT_ERROR_VALUE, config));
            }            
        }                       
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kEndFinallyEx();
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceSelect_OutputValue(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}
