/**
 * @file    TestSurfaceFeatures.c
 * @brief   Tool to create a feature line and point.
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestSurfaceFeatures.h>
#include <kVision/Common/kNullMath.h>
#include <math.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

kBeginClassEx(Tool, TestSurfaceFeatures)
    kAddVMethod(TestSurfaceFeatures, kObject, VRelease)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VInit)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VName)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VDescribe)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VNewMeasurementConfig)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VUpdateConfig)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VNewToolConfig)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VStart)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VStop)
    kAddVMethod(TestSurfaceFeatures, GdkTool, VProcess)
kEndClassEx()

/////////////////////////////////////////////////////////////////////////////
// GtTool functions
/////////////////////////////////////////////////////////////////////////////

ToolFx(const kChar*) TestSurfaceFeatures_VName()
{
    return GDK_SURFACE_FEATURES_TEST_TOOL_NAME;
}

ToolFx(kStatus) TestSurfaceFeatures_VDescribe(GdkToolInfo toolInfo)
{
    GdkOutputInfo outInfo = kNULL;
    k64f defFloat = 0.0;

    kCheck(GdkToolInfo_SetLabel(toolInfo, GDK_SURFACE_FEATURES_TEST_TOOL_LABEL));
    kCheck(GdkToolInfo_SetTypeName(toolInfo, GDK_SURFACE_FEATURES_TEST_TOOL_NAME));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE)); // Change this line to support other modes
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    // Input Parameters
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_0, GDK_SURFACE_FEATURES_TEST_PARAM_0, &defFloat, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_1, GDK_SURFACE_FEATURES_TEST_PARAM_1, &defFloat, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_2, GDK_SURFACE_FEATURES_TEST_PARAM_2, &defFloat, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_3, GDK_SURFACE_FEATURES_TEST_PARAM_3, &defFloat, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_4, GDK_SURFACE_FEATURES_TEST_PARAM_4, &defFloat, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, GDK_SURFACE_FEATURES_TEST_PARAM_5, GDK_SURFACE_FEATURES_TEST_PARAM_5, &defFloat, kNULL));

    // Ouput Features
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_FEATURE_LINE,  "LineOut",  "Line Output",  &outInfo));
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_FEATURE_POINT, "PointOut", "Point Output", &outInfo));

    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VInit(TestSurfaceFeatures tool, kType type, kAlloc alloc)
{
    kObjR(TestSurfaceFeatures, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    kZero(obj->lineDirection);
    kZero(obj->point);

    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VRelease(TestSurfaceFeatures tool)
{
    kObj(TestSurfaceFeatures, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestSurfaceFeatures_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VStart(TestSurfaceFeatures tool)
{
    kObj(TestSurfaceFeatures, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);

    obj->dataSource = GdkToolCfg_Source(config);

    obj->point.x = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_0));
    obj->point.y = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_1));
    obj->point.z = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_2));

    obj->lineDirection.x = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_3));
    obj->lineDirection.y = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_4));
    obj->lineDirection.z = GdkParam_AsFloat(GdkParams_Find(params, GDK_SURFACE_FEATURES_TEST_PARAM_5));

    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VStop(TestSurfaceFeatures tool)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_VProcess(TestSurfaceFeatures tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestSurfaceFeatures, tool);

    kTry
    {
        // Send Features
        kTest(TestSurfaceFeatures_SendFeatureLine(tool, output, 0, &obj->point, &obj->lineDirection));
        kTest(TestSurfaceFeatures_SendFeaturePoint(tool, output, 1, &obj->point));
    }
    kFinally
    {
        kEndFinally();
    }

    return kOK;
}

/////////////////////////////////////////////////////////////////////////////
// Output rendering
/////////////////////////////////////////////////////////////////////////////

ToolFx(kStatus) TestSurfaceFeatures_OutputValue(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_SendFeatureLine(TestSurfaceFeatures tool, GdkToolOutput output, kSize index,
    kPoint3d64f* linePosition, kPoint3d64f* lineDirection)
{
    GvLineFeatureMsg msg = kNULL;

    kCheck(GdkToolOutput_InitFeatureAt(output, index, &msg));
    if (msg != kNULL)
    {
        kCheck(GvFeatureMsg_SetPosition(msg, linePosition));
        kCheck(GvLineFeatureMsg_SetDirection(msg, lineDirection));
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceFeatures_SendFeaturePoint(TestSurfaceFeatures tool, GdkToolOutput output, kSize index, const kPoint3d64f* ptPosition)
{
    GvPointFeatureMsg msg = kNULL;

    kCheck(GdkToolOutput_InitFeatureAt(output, index, &msg));
    if (msg != kNULL)
    {
        kCheck(GvFeatureMsg_SetPosition(msg, ptPosition));
    }

    return kOK;
}
