/**
 * @file    TestSurfaceConfiguration.c
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestSurfaceConfiguration.h>
#include <math.h>

kBeginClassEx(Tool, TestSurfaceConfiguration)
    kAddVMethod(TestSurfaceConfiguration, kObject, VRelease)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VInit)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VDescribe)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VUpdateConfig)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VStart)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VStop)
    kAddVMethod(TestSurfaceConfiguration, GdkTool, VProcess)
kEndClassEx()

ToolFx(kStatus) TestSurfaceConfiguration_VDescribe(GdkToolInfo toolInfo)
{
    GdkParamInfo paramInfo;
    GdkMeasurementInfo mmtInfo;
    kBool defBool = kTRUE;
    k32s defInt = 0;
    GdkRegion3d64f defRegion = { -4, -3, -2, 2, 2, 4 };
    k64f defFloat1 = 1.0, defFloat0 = 0.0;

    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_SURFACE_CONFIGURATION_TOOL_NAME));
    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_SURFACE_CONFIGURATION_TOOL_LABEL));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Y, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    // Region input
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_BOOL, "UseRegion", "Use Region", &defBool, &paramInfo));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION, "Region", "Region", &defRegion, &paramInfo));
    kCheck(GdkParamInfo_EnableSurfaceRegionZAngle(paramInfo));

    // Drop down list
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, "InputValue", "Value", &defFloat1, &paramInfo)); // Set default value to 1.0

    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_INT, "Operator", "Operator", &defInt, &paramInfo));
    kCheck(GdkParamInfo_AddOptionInt(paramInfo, TEST_SURFACE_CONFIGURATION_OPERATOR_SQUARE, "Square"));
    kCheck(GdkParamInfo_AddOptionInt(paramInfo, TEST_SURFACE_CONFIGURATION_OPERATOR_SQRT, "Square Root"));
    kCheck(GdkParamInfo_AddOptionInt(paramInfo, TEST_SURFACE_CONFIGURATION_OPERATOR_SCALE, "Scale"));

    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT, "Factor", "Factor", &defFloat1, &paramInfo)); // Set default value to 1.0

    // Create measurement. Must the same order as measurement index!
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Result", "Result", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z));

    // This is an optional parameter. To test, save a recording with this line commented out then load the recording with this line in
    // Also this is a measurement level parameter
    kCheck(GdkOutputInfo_AddParam(mmtInfo, GDK_PARAM_TYPE_FLOAT, "Offset", "Offset", &defFloat0, &paramInfo));
    kCheck(GdkParamInfo_SetIsOptional(paramInfo, kTRUE));
    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_VInit(TestSurfaceConfiguration tool, kType type, kAlloc alloc)
{
    kObjR(TestSurfaceConfiguration, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    obj->useRegion = kFALSE;
    kZero(obj->region);
    obj->inputValue = 0.0;
    obj->surfaceOperator = TEST_SURFACE_CONFIGURATION_OPERATOR_SQUARE;
    obj->factor = 0.0;
    obj->measurementOffset = 0.0;

    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_VRelease(TestSurfaceConfiguration tool)
{
    kObj(TestSurfaceConfiguration, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestSurfaceConfiguration_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    GdkParams params = GdkToolCfg_Parameters(toolConfig);

    // Enable region is we are using it
    GdkParam paramUseRegion = GdkParams_Find(params, "UseRegion");
    GdkParam paramRegion = GdkParams_Find(params, "Region");
    GdkParam paramOperator = GdkParams_Find(params, "Operator");
    GdkParam paramFactor = GdkParams_Find(params, "Factor");

    // Enable region input if the boolean is checked
    kCheck(GdkParam_SetUsed(paramRegion, GdkParam_AsBool(paramUseRegion)));

    // Show factor parameter if operator is scaling
    if (GdkParam_AsInt(paramOperator) == TEST_SURFACE_CONFIGURATION_OPERATOR_SCALE)
    {
        kCheck(GdkParam_SetUsed(paramFactor, kTRUE));
    }
    else
    {
        kCheck(GdkParam_SetUsed(paramFactor, kFALSE));
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_VStart(TestSurfaceConfiguration tool)
{
    kObj(TestSurfaceConfiguration, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkMeasurementCfg measurementConfig = GdkToolCfg_MeasurementAt(config, TEST_SURFACE_CONFIGURATION_RESULT);

    GdkParam param;

    obj->dataSource = GdkToolCfg_Source(config);

    // Loading tools parameeters
    param = GdkParams_Find(params, "UseRegion");
    obj->useRegion = GdkParam_AsBool(param);

    param = GdkParams_Find(params, "Region");
    obj->region = *GdkParam_AsSurfaceRegion(param);

    param = GdkParams_Find(params, "InputValue");
    obj->inputValue = GdkParam_AsFloat(param);

    param = GdkParams_Find(params, "Operator");
    obj->surfaceOperator = (TestSurfaceConfigurationOperator) GdkParam_AsInt(param);

    param = GdkParams_Find(params, "Factor");
    obj->factor = GdkParam_AsFloat(param);

    // Loading measurement parameter
    param = GdkParams_Find(GdkMeasurementCfg_Parameters(measurementConfig), "Offset");
    obj->measurementOffset = GdkParam_AsFloat(param);
    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_VStop(TestSurfaceConfiguration tool)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_OutputValue(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceConfiguration_VProcess(TestSurfaceConfiguration tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestSurfaceConfiguration, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkInputItem item;
    kStatus exception = kOK;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item) return kERROR_PARAMETER;

    kTry
    {
        switch (obj->surfaceOperator)
        {
            case TEST_SURFACE_CONFIGURATION_OPERATOR_SQUARE:
            {
                kTest(TestSurfaceConfiguration_OutputValue(output, TEST_SURFACE_CONFIGURATION_RESULT, (obj->inputValue * obj->inputValue), GDK_MEASUREMENT_OK, config));
                break;
            }
            case TEST_SURFACE_CONFIGURATION_OPERATOR_SQRT:
            {
                if (obj->inputValue >= 0)
                {
                    kTest(TestSurfaceConfiguration_OutputValue(output, TEST_SURFACE_CONFIGURATION_RESULT, sqrt(obj->inputValue), GDK_MEASUREMENT_OK, config));
                }
                else
                {
                    kTest(TestSurfaceConfiguration_OutputValue(output, TEST_SURFACE_CONFIGURATION_RESULT, 0.0, GDK_MEASUREMENT_ERROR_VALUE, config));
                }
                break;
            }
            case TEST_SURFACE_CONFIGURATION_OPERATOR_SCALE:
            {

                kTest(TestSurfaceConfiguration_OutputValue(output, TEST_SURFACE_CONFIGURATION_RESULT, (obj->inputValue * obj->factor), GDK_MEASUREMENT_OK, config));
                break;
            }
        }
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kEndFinallyEx();
    }

    return kOK;
}

