/**
* @file    GdkFile.h
* @brief   Declares the GdkFile class.
*
* Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_FILE_H
#define GDK_FILE_H

#include <Gdk/GdkDef.h>

/**
* Describes the interfaces to perform file operations from a Gdk tool.
*
* @class       GdkFile
* @ingroup     Gdk-Utils
*/

/**
* Read a file
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   fileName     Filename. (Name of the file to read from. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @param   data         Receives a pointer to a kArray1 containing the data.
* @return               Operation status.
*/
GdkFx(kStatus) GdkFile_Read(const kChar *fileName, kArray1* data);

/**
* Write a file
*
* Do not write to files too often. (More than a few times per sensor startup)
* Sensor flash memory may be corrupted by excessive writes.
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   fileName     Filename. (Name of the file to be written to. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @param   data         Pointer to data to write to file.
* @param   size         Size of data to write to file.
* @return               Operation status.
*/
GdkFx(kStatus) GdkFile_Write(const kChar *fileName, const void* data, kSize size);

/**
* How much flash memory is available to write to.
*
* @public               @memberof GdkFile
* @return               Available (bytes).
*/
GdkFx(k64u) GdkFile_UserStorageFree();

/**
* How much flash memory has been used.
*
* @public               @memberof GdkFile
* @return               Used (bytes).
*/
GdkFx(k64u) GdkFile_UserStorageUsed();

/**
* Delete the file. 
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   fileName     Filename. (Name of the file to be deleted. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @return               Operation status.
*/
GdkFx(kStatus) GdkFile_Delete(const kChar *fileName);

/**
* Does the file exist within the user directory.
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   fileName     Filename. (Name of the file to determine existance. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @return               File existance.
*/
GdkFx(kBool) GdkFile_Exists(const kChar *fileName);

/**
* Rename the file.
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   from         Current filename. (Name of the file to rename. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @param   to           To be renamed filename. (Name of the new file. Name must include "user/" in front of the file name. For example, "user/foonew" to access file "foonew").
* @return               Operation status.
*/
GdkFx(kStatus) GdkFile_Rename(const kChar *from, const kChar *to);

/**
* Length of file - Query the file length (number of bytes in file). 
*
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VProcess CALLBACK
* DO NOT CALL THIS FUNCTION FROM A GDK TOOL_VStop CALLBACK
*
* @public               @memberof GdkFile
* @param   fileName     Filename. (Name of the file to be queried for length. Name must include "user/" in front of the file name. For example, "user/foo" to access file "foo").
* @return               Length (bytes).
*/
GdkFx(k64u) GdkFile_Length(const kChar *fileName);

#endif
