/**
* @file    GdkToolVersionInfo.h
* @brief   Declares the GdkToolVersionInfo class.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_TOOL_VERSION_INFO_H
#define GDK_TOOL_VERSION_INFO_H

#include <Gdk/GdkDef.h>
#include <Gdk/Tools/GdkMeasurementInfo.h>
#include <Gdk/Tools/GdkFeatureInfo.h>
#include <Gdk/Tools/GdkToolDataOutputInfo.h>
#include <Gdk/Config/GdkParamsInfo.h>

/**
* Describes a tool config version. A version consists of a set of parameters, measurements,
* and the parameters of those measurements.
*
* @class       GdkToolVersionInfo
* @extends     kObject
* @ingroup     Gdk-Tools
*/
typedef kObject GdkToolVersionInfo;
kDeclareClassEx(Gdk, GdkToolVersionInfo, kObject)

/**
* Use another version as the base for this version. All parameters and measurements
* are copied.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    source      Source info object.
* @return               Params info.
*/
GdkFx(kStatus) GdkToolVersionInfo_UseBase(GdkToolVersionInfo info, GdkToolVersionInfo source);

/**
* Enable or disable auto versioning. When disabled, the version is set to "1" if not already set.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    enable      Enable or disable.
* @return               Params info.
*/
GdkFx(kStatus) GdkToolVersionInfo_EnableAutoVersion(GdkToolVersionInfo info, kBool enable);

/**
* Returns the schema for the parameters.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Params info.
*/
GdkFx(GdkParamsInfo) GdkToolVersionInfo_Params(GdkToolVersionInfo info);

/**
* Add a measurement type.
*
* @public                   @memberof GdkToolVersionInfo
* @param    info            Info object.
* @param    type            Measurement type name (should be unique within the tool).
* @param    measurementInfo Pointer to receive the new measurement info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddMeasurement(GdkToolVersionInfo info, const kChar* type, GdkMeasurementInfo* measurementInfo);

/**
* Clears measurement types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_ClearMeasurements(GdkToolVersionInfo info);

/**
* Returns the number of measurement types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Measurement type count.
*/
GdkFx(kSize) GdkToolVersionInfo_MeasurementCount(GdkToolVersionInfo info);

/**
* Returns the measurement type at a specified index.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    index       Measurement type index.
* @return               Measurement type.
*/
GdkFx(GdkMeasurementInfo) GdkToolVersionInfo_MeasurementAt(GdkToolVersionInfo info, kSize index);

/**
* Add a feature type.
*
* @public                   @memberof GdkToolVersionInfo
* @param    info            Info object.
* @param    type            Feature type. 
* @param    dataType        Type of feature data (Line, Circle, etc)
* @param    featureInfo Pointer to receive the new feature info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddFeature(GdkToolVersionInfo info, const kChar* type, GdkFeatureDataType dataType, GdkFeatureInfo* featureInfo);

/**
* Clears feature types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_ClearFeatures(GdkToolVersionInfo info);

/**
* Returns the number of feature types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Feature type count.
*/
GdkFx(kSize) GdkToolVersionInfo_FeatureCount(GdkToolVersionInfo info);

/**
* Returns the feature type at a specified index.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    index       Feature type index.
* @return               Feature type.
*/
GdkFx(GdkFeatureInfo) GdkToolVersionInfo_FeatureAt(GdkToolVersionInfo info, kSize index);

/**
* Add a tool data output type.
*
* @public                   @memberof GdkToolVersionInfo
* @param    info            Info object.
* @param    type            Type of the tool data output.
* @param    dataType        Type of data:
*                           - #GDK_DATA_TYPE_RANGE
*                           - #GDK_DATA_TYPE_UNIFORM_PROFILE
*                           - #GDK_DATA_TYPE_PROFILE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_UNIFORM_SURFACE
*                           - #GDK_DATA_TYPE_SURFACE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_MESH
* @param    toolDataOutputInfo  Pointer to receive the new tool data output info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddToolDataOutput(GdkToolVersionInfo info, const kChar* type, GdkDataType dataType, GdkToolDataOutputInfo* toolDataOutputInfo);

/**
* Clears tool data output types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_ClearToolDataOutputs(GdkToolVersionInfo info);

/**
* Returns the number of tool data output types.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Tool data output type count.
*/
GdkFx(kSize) GdkToolVersionInfo_ToolDataOutputCount(GdkToolVersionInfo info);

/**
* Returns the tool data output type at a specified index.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    index       Tool data output type index.
* @return               Tool data output type.
*/
GdkFx(GdkToolDataOutputInfo) GdkToolVersionInfo_ToolDataOutputAt(GdkToolVersionInfo info, kSize index);

/**
* Sets the version name. If the name is specified as kNULL, a version will
* automatically be generated based on the contents of the version object.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @param    version     Version name (can be empty string or kNULL).
* @return               Found version object or kNULL.
*/
GdkFx(kStatus) GdkToolVersionInfo_SetName(GdkToolVersionInfo info, const kChar* version);

/**
* Returns the version name.
*
* @public               @memberof GdkToolVersionInfo
* @param    info        Info object.
* @return               Version name.
*/
GdkFx(const kChar*) GdkToolVersionInfo_Name(GdkToolVersionInfo info);

/**
* Adds a data output to a tool with the specified data type

* @public                   @memberof GdkToolVersionInfo
* @param    versionInfo     Version info object.
* @param    dataType        Tool data output type. The following types are supported:
*                           - #GDK_DATA_TYPE_MEASUREMENT
*                           - #GDK_DATA_TYPE_FEATURE_POINT
*                           - #GDK_DATA_TYPE_FEATURE_LINE
*                           - #GDK_DATA_TYPE_FEATURE_CIRCLE
*                           - #GDK_DATA_TYPE_FEATURE_PLANE
*                           - #GDK_DATA_TYPE_UNIFORM_PROFILE
*                           - #GDK_DATA_TYPE_PROFILE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_UNIFORM_SURFACE
*                           - #GDK_DATA_TYPE_SURFACE_POINT_CLOUD
*                           - #GDK_DATA_TYPE_MESH
*                           - #GDK_DATA_TYPE_GENERIC_1 to N
* @param    name            Name of the output type.
* @param    label           Label of the output.
* @param    outputInfo      Pointer to receive the new data output's info object.
*                           It is based on the dataType and should be one of the following:
*                               GdkMeasurementInfo
*                               GdkFeatureInfo
*                               GdkToolDataOutputInfo
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddOutput(GdkToolVersionInfo versionInfo, GdkDataType dataType, const kChar* name, const kChar* label, GdkOutputInfo *outputInfo);

/**
* Adds an input data set to a tool with the specified data type

* @public                   @memberof GdkToolVersionInfo
* @param    versionInfo     Version info object.
* @param    dataType        Tool data output type. The following types are supported:
*                           - #GDK_DATA_TYPE_FEATURE_POINT
*                           - #GDK_DATA_TYPE_FEATURE_LINE
*                           - #GDK_DATA_TYPE_FEATURE_CIRCLE
*                           - #GDK_DATA_TYPE_FEATURE_PLANE
* @param    name            Name of the input.
* @param    label           Label of the input. If kNULL, will be set to same as name parameter
* @param    paramInfo       Pointer to receive the new intput data set's paramInfo object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddInput(GdkToolVersionInfo versionInfo, GdkDataType dataType, const kChar* name, const kChar* label, GdkParamInfo *paramInfo);

/**
* Adds a parameter to a tool

* @public                   @memberof GdkToolVersionInfo
* @param    versionInfo     Version info object.
* @param    name            Name of the parameter.
* @param    paramType       Tool parameter type. The following types are supported:
*                           - #GDK_PARAM_TYPE_INT
*                           - #GDK_PARAM_TYPE_FLOAT
*                           - #GDK_PARAM_TYPE_BOOL
*                           - #GDK_PARAM_TYPE_PROFILE_REGION
*                           - #GDK_PARAM_TYPE_SURFACE_REGION
*                           - #GDK_PARAM_TYPE_SURFACE_REGION_2D
* @param    label           Label of the param. If kNULL, will be set to same as name parameter
* @param    defValue        Pointer to the default value to set the parameter. This depends on the paramType passed.
* @param    paramInfo       Pointer to receive the new input data set's paramInfo object.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolVersionInfo_AddParam(GdkToolVersionInfo versionInfo, GdkParamType paramType, const kChar* name, const kChar* label, const void* defValue, GdkParamInfo *paramInfo);

#endif 
